package com.amity.socialcloud.sdk.social.domain.story.create

import android.net.Uri
import com.amity.socialcloud.sdk.core.data.file.FileRepository
import com.amity.socialcloud.sdk.model.core.content.AmityContentFeedType
import com.amity.socialcloud.sdk.model.core.file.AmityVideo
import com.amity.socialcloud.sdk.model.core.file.upload.AmityUploadResult
import com.amity.socialcloud.sdk.model.social.story.AmityStory
import com.amity.socialcloud.sdk.model.social.story.AmityStoryItem
import com.amity.socialcloud.sdk.social.data.story.StoryRepository
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single

class CreateVideoStoryUseCase : CreateStoryUseCase {

    fun execute(
        targetType: AmityStory.TargetType,
        targetId: String,
        fileUri: Uri,
        storyItems: List<AmityStoryItem>,
        metadata: JsonObject?
    ): Completable {
        return createStory(
            targetType = targetType,
            targetId = targetId,
            createLocalFile = { uniqueId ->
                createLocalFile(
                    uniqueId = uniqueId,
                    fileUri = fileUri
                )
            },
            createLocalStory = { uniqueId ->
                createLocalStory(
                    uniqueId = uniqueId,
                    targetType = targetType,
                    targetId = targetId,
                    storyItems = storyItems,
                    metadata = metadata
                )
            },
            uploadFile = { uniqueId ->
                uploadFile(
                    uniqueId,
                    fileUri
                )
            },
            createRemoteStory = { uniqueId, fileId ->
                createRemoteStory(
                    uniqueId = uniqueId,
                    targetType = targetType,
                    targetId = targetId,
                    fileId = fileId,
                    storyItems = storyItems,
                    metadata = metadata
                )
            }
        )
    }

    private fun createLocalFile(
        uniqueId: String,
        fileUri: Uri,
    ): Completable {
        return FileRepository().createLocalFile(
            fileId = uniqueId,
            fileType = AmityStory.DataType.VIDEO.apiKey,
            filePath = fileUri.toString()
        )
    }

    private fun createLocalStory(
        uniqueId: String,
        targetType: AmityStory.TargetType,
        targetId: String,
        storyItems: List<AmityStoryItem> = emptyList(),
        metadata: JsonObject?
    ): Completable {
        return StoryRepository().createLocalVideoStory(
            referenceId = uniqueId,
            targetType = targetType,
            targetId = targetId,
            storyItems = storyItems,
            metadata = metadata,
        )
    }

    private fun uploadFile(
        uniqueId: String,
        fileUri: Uri
    ): Flowable<AmityUploadResult<AmityVideo>> {
        return FileRepository().uploadVideo(
            uploadId = uniqueId,
            uri = fileUri,
            feedType = AmityContentFeedType.STORY
        )
    }

    private fun createRemoteStory(
        uniqueId: String,
        targetType: AmityStory.TargetType,
        targetId: String,
        fileId: String,
        storyItems: List<AmityStoryItem> = emptyList(),
        metadata: JsonObject?
    ): Single<String> {
        return StoryRepository().createVideoStory(
            referenceId = uniqueId,
            targetType = targetType,
            targetId = targetId,
            fileId = fileId,
            storyItems = storyItems,
            metadata = metadata
        )
    }

}