package com.amity.socialcloud.sdk.social.data.community

import com.amity.socialcloud.sdk.dto.community.JoinRequestQueryDto
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.amity.socialcloud.sdk.model.social.community.AmityCommunityMembershipStatusFilter
import com.amity.socialcloud.sdk.model.social.community.AmityCommunityStorySettings
import com.ekoapp.ekosdk.internal.api.AmityHttpClient
import com.ekoapp.ekosdk.internal.api.dto.CommunityListQueryDto
import com.ekoapp.ekosdk.internal.api.dto.EkoDeletionDto
import com.ekoapp.ekosdk.internal.api.socket.request.CommunityCreateRequest
import com.ekoapp.ekosdk.internal.api.socket.request.CommunityUpdateRequest
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Single

internal class CommunityRemoteDataStore {

    fun createCommunity(
        displayName: String,
        description: String?,
        categoryIds: List<String>?,
        isPublic: Boolean?,
        metadata: JsonObject?,
        userIds: List<String>?,
        avatarFileId: String?,
        needApprovalOnPostCreation: Boolean?,
        onlyAdminCanPost: Boolean?,
        storySettings: AmityCommunityStorySettings?,
        isDiscoverable: Boolean? = null,
        requiresJoinApproval: Boolean? = null,
    ): Single<CommunityListQueryDto> {
        val request = CommunityCreateRequest(
            displayName = displayName,
            description = description,
            categoryIds = categoryIds,
            isPublic = isPublic,
            metadata = metadata,
            userIds = userIds,
            avatarFileId = avatarFileId,
            needApprovalOnPostCreation = needApprovalOnPostCreation,
            onlyAdminCanPost = onlyAdminCanPost,
            allowCommentInStory = storySettings?.allowComment,
            isDiscoverable = isDiscoverable,
            requiresJoinApproval = requiresJoinApproval
        )

        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.createCommunity(request)
        }
    }

    fun updateCommunity(
        communityId: String,
        displayName: String?,
        description: String?,
        categoryIds: List<String>?,
        isPublic: Boolean?,
        metadata: JsonObject?,
        avatarFileId: String?,
        needApprovalOnPostCreation: Boolean?,
        onlyAdminCanPost: Boolean?,
        storySettings: AmityCommunityStorySettings?,
        isDiscoverable: Boolean? = null,
        requiresJoinApproval: Boolean? = null,
    ): Single<CommunityListQueryDto> {
        val request = CommunityUpdateRequest(
            communityId = communityId,
            displayName = displayName,
            description = description,
            categoryIds = categoryIds,
            isPublic = isPublic,
            metadata = metadata,
            avatarFileId = avatarFileId,
            needApprovalOnPostCreation = needApprovalOnPostCreation,
            onlyAdminCanPost = onlyAdminCanPost,
            allowCommentInStory = storySettings?.allowComment,
            isDiscoverable = isDiscoverable,
            requiresJoinApproval = requiresJoinApproval
        )
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.updateCommunity(communityId, request)
        }
    }

    fun joinCommunity(communityId: String): Single<CommunityListQueryDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.joinCommunity(communityId)
        }
    }

    fun getJoinRequest(
        communityId: String,
        status: String,
        sortBy: String?,
        targetType: String,
        type: String,
        limit: Int? = null,
        token: String? = null,
    ): Single<JoinRequestQueryDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.getJoinRequest(
                communityId = communityId,
                status = status,
                sortBy = sortBy,
                targetType = targetType,
                type = type,
                limit = limit,
                token = token
            )
        }
    }

    fun getLatestJoinRequest(
        communityIds: List<String>,
    ): Single<JoinRequestQueryDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.getLatestJoinRequest(
                communityIds = communityIds,
            )
        }
    }

    fun getMyJoinRequest(
        communityId: String,
    ): Single<JoinRequestQueryDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.getMyJoinRequest(
                communityId = communityId,
            )
        }
    }

    fun joinRequest(communityId: String): Single<JoinRequestQueryDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.joinRequest(communityId)
        }
    }

    fun approveJoinRequest(communityId: String, userId: String): Single<EkoDeletionDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.approveJoinRequest(communityId, userId)
        }
    }

    fun rejectJoinRequest(communityId: String, userId: String): Single<EkoDeletionDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.rejectJoinRequest(communityId, userId)
        }
    }

    fun cancelJoinRequest(communityId: String): Single<EkoDeletionDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.cancelJoinRequest(communityId)
        }
    }

    fun leaveCommunity(communityId: String): Single<CommunityListQueryDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.leaveCommunity(communityId)
        }
    }

    fun deleteCommunity(communityId: String): Single<EkoDeletionDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.deleteCommunity(communityId)
        }
    }

    fun getCommunity(communityId: String): Single<CommunityListQueryDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.getCommunity(communityId)
        }
    }

    fun getRecommendedCommunities(
        options: QueryOptionsRequestParams,
        includeDiscoverablePrivateCommunity: Boolean,
    ): Single<CommunityListQueryDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.getRecommendedCommunities(
                includeDiscoverablePrivateCommunity = includeDiscoverablePrivateCommunity,
                limit = options.limit,
                token = options.token
            )
        }
    }

    fun getTrendingCommunities(
        options: QueryOptionsRequestParams,
        includeDiscoverablePrivateCommunity: Boolean,
    ): Single<CommunityListQueryDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.getTrendingCommunities(
                includeDiscoverablePrivateCommunity = includeDiscoverablePrivateCommunity,
                limit = options.limit,
                token = options.token
            )
        }
    }

    fun queryCommunities(
        keyword: String? = null,
        includeDiscoverablePrivateCommunity: Boolean,
        categoryId: String? = null,
        filter: String? = null,
        sortBy: String? = null,
        isDeleted: Boolean? = null,
        options: QueryOptionsRequestParams,
    ): Single<CommunityListQueryDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.queryCommunities(
                keyword = keyword,
                includeDiscoverablePrivateCommunity = includeDiscoverablePrivateCommunity,
                categoryId = categoryId,
                filter = filter,
                sortBy = sortBy,
                isDeleted = isDeleted,
                limit = options.limit,
                token = options.token,
            )
        }
    }

    fun searchCommunities(
        query: String,
        filter: AmityCommunityMembershipStatusFilter?,
        tags: List<String>? = null,
        categoryIds: List<String>? = null,
        includeDiscoverablePrivateCommunity: Boolean,
        limit: Int? = null,
        token: String? = null,
    ): Single<CommunityListQueryDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.searchCommunities(
                query = query,
                filter = filter?.value,
                includeDiscoverablePrivateCommunity = includeDiscoverablePrivateCommunity,
                tags = tags?.filter { it.isNotBlank() }.takeIf { it?.isNotEmpty() == true },
                categoryIds = categoryIds?.filter { it.isNotBlank() }
                    .takeIf { it?.isNotEmpty() == true },
                limit = limit,
                token = token
            )
        }
    }

    fun getCommunityByIds(communityIds: List<String>, includeDiscoverablePrivateCommunity: Boolean? = null): Single<CommunityListQueryDto> {
        return AmityHttpClient.get(CommunityApi::class).flatMap {
            it.getCommunityByIds(communityIds)
        }
    }

}