package com.amity.socialcloud.sdk.core.data.reaction.paging

import androidx.paging.ExperimentalPagingApi
import co.amity.rxbridge.toRx2
import com.amity.socialcloud.sdk.core.data.invitation.InvitationQueryPersister
import com.amity.socialcloud.sdk.core.data.invitation.InvitationRemoteDataStore
import com.amity.socialcloud.sdk.model.core.invitation.AmityInvitation
import com.amity.socialcloud.sdk.model.core.invitation.AmityInvitationType
import com.amity.socialcloud.sdk.social.data.community.CommunityRepository
import com.ekoapp.ekosdk.internal.api.dto.InvitationsQueryDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.data.model.InvitationEntity
import com.ekoapp.ekosdk.internal.keycreator.InvitationKeyCreator
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import com.ekoapp.ekosdk.internal.token.QueryStreamQueryToken
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single

@OptIn(ExperimentalPagingApi::class)
class InvitationMediator(
    val type: AmityInvitationType,
    val target: Target,
) :
    DynamicQueryStreamMediator<InvitationEntity, InvitationsQueryDto, AmityInvitation>(
        dynamicQueryStreamKeyCreator = InvitationKeyCreator(
            type = type,
            target = target
        ),
        nonce = AmityNonce.INVITATION_LIST
    ) {

    sealed class Target(val type: String, val id: String) {
        object ME : Target("me", "me")
        class COMMUNITY(val communityId: String) : Target("community", communityId)
    }

    override fun provideReactorPublisher(): Flowable<AmityInvitation> {
        return Flowable.empty()
    }

    override fun getFirstPageRequest(pageSize: Int): Single<InvitationsQueryDto> {
        return getRequest(
            limit = pageSize
        )
    }

    override fun getFetchByTokenRequest(token: String): Single<InvitationsQueryDto> {
        return getRequest(
            token = token
        )
    }

    override fun persistResponse(dto: InvitationsQueryDto): Completable {
        val communityIds = dto.invitations
                .filter { it.targetType == "community" }
                .mapNotNull { it.targetId }
        return InvitationQueryPersister().persist(dto)
                .andThen(CommunityRepository().fetchCommunityByIds(communityIds))
                .toRx2()
    }

    override fun convertResponseToQueryToken(dto: InvitationsQueryDto): QueryStreamQueryToken {
        val ids = mutableListOf<String>()
        dto.invitations.forEach { invitationDto ->
            ids.add(invitationDto.invitationId)
        }
        return DynamicQueryStreamQueryToken(
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            next = dto.token?.next,
            previous = dto.token?.previous,
            primaryKeys = ids,
        )
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<InvitationsQueryDto> {
        return if (target is Target.ME) {
            InvitationRemoteDataStore().getMyInvitations(
                type = type,
                limit = limit,
                token = token
            ).toRx2()
        } else {
            InvitationRemoteDataStore().getInvitations(
                type = type,
                target = target,
                limit = limit,
                token = token
            ).toRx2()
        }
    }
}