package com.amity.socialcloud.sdk.core.session.component

import android.util.Log
import com.amity.socialcloud.sdk.core.domain.session.GetCurrentAccountUseCase
import com.amity.socialcloud.sdk.core.session.TokenVerifier
import com.amity.socialcloud.sdk.core.session.eventbus.AppEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.core.session.model.AppEvent
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.disposables.CompositeDisposable
import io.reactivex.rxjava3.schedulers.Schedulers
import java.util.concurrent.TimeUnit

class TokenWatcherSessionComponent(
    private val appEventBus: AppEventBus,
    sessionLifeCycleEventBus: SessionLifeCycleEventBus,
    sessionStateEventBus: SessionStateEventBus
) : SessionComponent(
    sessionLifeCycleEventBus, sessionStateEventBus
) {

    var compositeDisposable: CompositeDisposable = CompositeDisposable()

    override fun onSessionStateChange(sessionState: SessionState) {

    }

    override fun establish(account: EkoAccount) {
        disposeSubscription()
        compositeDisposable = CompositeDisposable()
        observeSessionState()
        observeAppEvent()
        startTokenWatcherInterval()
    }

    override fun destroy() {
        disposeSubscription()
    }

    override fun handleTokenExpire() {
        disposeSubscription()
    }

    private fun checkAccessTokenExpiration() {
        GetCurrentAccountUseCase()
            .execute()
            .doOnSuccess {
                val tokenResult = TokenVerifier.checkExpiration(it)
                Log.e("SSM3", "token verification result: $tokenResult")
                if (tokenResult == TokenVerifier.Result.ABOUT_TO_EXPIRE) {
                    appEventBus.publish(AppEvent.TokenAboutToExpire)
                } else if (tokenResult == TokenVerifier.Result.EXPIRED) {
                    appEventBus.publish(AppEvent.TokenExpire)
                }
            }
            .subscribeOn(Schedulers.io())
            .subscribe()
    }

    //check access token expiration when a user continues/resumes an SDK session
    private fun observeSessionState() {
        val sessionStateDisposable = getSessionStateEventBus()
            .observe()
            .doOnNext { sessionState ->
                if (sessionState == SessionState.Established) {
                    checkAccessTokenExpiration()
                }
            }
            .subscribeOn(Schedulers.io())
            .subscribe()
        compositeDisposable.add(sessionStateDisposable)
    }

    //check access token expiration when a user continues/resumes an application session
    private fun observeAppEvent() {
        val appEventDisposable = appEventBus
            .observe()
            .filter { it is AppEvent.AppForeground }
            .doOnNext {
                checkAccessTokenExpiration()
            }
            .subscribeOn(Schedulers.io())
            .subscribe()
        compositeDisposable.add(appEventDisposable)
    }

    //check access token expiration automatically every 10 minutes regardless of user/sdk events
    private fun startTokenWatcherInterval() {
        // FIXME: change seconds to minutes
        val tokenIntervalDisposable = Flowable.interval(1L, TimeUnit.MINUTES, Schedulers.io())
            .doOnNext {
                checkAccessTokenExpiration()
            }
            .subscribe()
        compositeDisposable.add(tokenIntervalDisposable)
    }

    private fun disposeSubscription() {
        compositeDisposable.dispose()
        compositeDisposable.clear()
    }
}