package com.amity.socialcloud.sdk.social.data.post.paging

import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.amity.socialcloud.sdk.social.data.post.PostRepositoryHelper
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator

class UserPostKeyCreator(
    private val userId: String,
    private val sortBy: String,
    private val isDeleted: Boolean?,
    private val postTypes: List<AmityPost.DataType>
) : DynamicQueryStreamKeyCreator {

    override fun toMap(): Map<String, Any> {
        return mapOf(
            "targetId" to userId,
            "targetType" to AmityPost.TargetType.USER.apiKey,
            "sortBy" to sortBy,
            "isDeleted" to (isDeleted ?: false),
            "postTypes" to PostRepositoryHelper.convertPostTypesToString(postTypes)
        )
    }

    override fun getFilterQuery(): String {
        return getConditionStatement(
            userId = userId,
            includeDeleted = isDeleted,
            postTypes = postTypes
        )
    }

    private fun getConditionStatement(
        userId: String,
        includeDeleted: Boolean?,
        postTypes: List<AmityPost.DataType>
    ): String {
        //userId filter
        val userFilterStatement = "post.targetType = '${AmityPost.TargetType.USER.apiKey}'" +
                " and post.targetId = '$userId'"

        //includeDelete filter
        var isDeletedFilterStatement = ""
        if (includeDeleted == false) {
            isDeletedFilterStatement = " and post.isDeleted = 0"
        } else if (includeDeleted == true) {
            isDeletedFilterStatement = " and post.isDeleted = 1"
        }
        //post types filter
        var postTypesFilterStatement = ""
        if (postTypes.isNotEmpty()) {
            postTypesFilterStatement =
                " and post.postDataType IN (${getPostTypesAsArrayString(postTypes)})"
        }
        //parentPost filter
        val parentPostFilter = if (postTypes.isNullOrEmpty()) {
            " and post.parentPostId is null"
        } else {
            " and post.targetId is not null"
        }

        //compose all condition
        return userFilterStatement +
                isDeletedFilterStatement +
                postTypesFilterStatement +
                parentPostFilter
    }

    fun getPostTypesAsArrayString(postTypes: List<AmityPost.DataType>): String {
        val typesString = StringBuilder()
        postTypes.forEachIndexed { index, dataType ->
            if (index == 0) {
                typesString.append("'${dataType.getApiKey()}'")
            } else {
                typesString.append(", '${dataType.getApiKey()}'")
            }
        }
        return typesString.toString()
    }

}