package com.amity.socialcloud.sdk.chat.data.channel.paging

import co.amity.rxbridge.toRx2
import com.amity.socialcloud.sdk.chat.data.channel.ChannelHelper
import com.amity.socialcloud.sdk.chat.data.channel.ChannelQueryPersister
import com.amity.socialcloud.sdk.chat.data.channel.ChannelRemoteDataStore
import com.amity.socialcloud.sdk.chat.data.channel.ChannelRepository
import com.amity.socialcloud.sdk.chat.data.marker.channel.ChannelMarkerRepository
import com.amity.socialcloud.sdk.core.MarkerSyncEngine
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.amity.socialcloud.sdk.model.chat.channel.AmityChannel
import com.amity.socialcloud.sdk.model.chat.channel.AmityChannelFilter
import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.ekoapp.ekosdk.internal.EkoChannelEntity
import com.ekoapp.ekosdk.internal.api.dto.ChannelQueryDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single


internal class ChannelMediator(
    private val types: Set<AmityChannel.Type>,
    private val filter: AmityChannelFilter,
    private val includingTags: AmityTags,
    private val excludingTags: AmityTags,
    private val isDeleted: Boolean?,
) : DynamicQueryStreamMediator<EkoChannelEntity, ChannelQueryDto, AmityChannel>(
    nonce = AmityNonce.CHANNEL_LIST,
    dynamicQueryStreamKeyCreator = ChannelKeyCreator(
        types = types,
        filter = filter,
        includingTags = includingTags,
        excludingTags = excludingTags,
        isDeleted = isDeleted
    )
) {

    override fun forceRefresh() = true

    override fun provideReactorPublisher(): Flowable<AmityChannel> {
        return ChannelRepository().getLatestChannel(
            types = types,
            filter = filter,
            includingTags = includingTags,
            excludingTags = excludingTags,
            isDeleted = isDeleted,
            dynamicQueryStreamKeyCreator =  dynamicQueryStreamKeyCreator,
            nonce = nonce
        ).toRx2()
    }

    override fun getFirstPageRequest(pageSize: Int): Single<ChannelQueryDto> {
        return getRequest(limit = pageSize)

    }

    override fun getFetchByTokenRequest(token: String): Single<ChannelQueryDto> {
        return getRequest(token = token)
    }

    override fun persistResponse(dto: ChannelQueryDto): Completable {
        return ChannelQueryDtoPersister().persistDto(dto).toRx2()
    }

    override fun convertResponseToQueryToken(dto: ChannelQueryDto): DynamicQueryStreamQueryToken {
        return DynamicQueryStreamQueryToken(
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            next = dto.token?.next,
            previous = dto.token?.previous,
            primaryKeys = dto.channelDtoList?.map { it.channelId } ?: emptyList()
        )
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<ChannelQueryDto> {
        return ChannelRemoteDataStore().queryChannels(
            types = ChannelHelper.getChannelTypes(types),
            filter = filter.apiKey,
            includingTags = includingTags,
            excludingTags = excludingTags,
            isDeleted = isDeleted,
            options = QueryOptionsRequestParams(limit, token)
        ).toRx2()
    }

}