package com.amity.socialcloud.sdk.core.data.notificationtray

import androidx.paging.PagingSource
import com.amity.socialcloud.sdk.entity.core.notificationtray.NotificationTrayItemEntity
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.ekoapp.ekosdk.internal.data.UserDatabase
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.schedulers.Schedulers
import org.joda.time.DateTime

internal class NotificationTrayLocalDataStore {

    fun observe(notificationId: String): Flowable<NotificationTrayItemEntity> {
        return UserDatabase.get().notificationTrayItemDao()
            .observeNotificationTrayItem(notificationId)
            .map {
                if (it == null) {
                    throw AmityException.create(
                        message = "item not found",
                        cause = null,
                        error = AmityError.ITEM_NOT_FOUND
                    )
                } else {
                    return@map it
                }
            }
    }

    fun getNotificationTrayItem(notificationId: String): NotificationTrayItemEntity? {
        var notificationTrayItem: NotificationTrayItemEntity? = null
        Completable.fromCallable {
            val notiItem = UserDatabase.get().notificationTrayItemDao().getByIdNow(notificationId)
            if (notiItem != null) {
                notificationTrayItem = notiItem
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return notificationTrayItem
    }

    fun saveNotificationTrayItems(items: List<NotificationTrayItemEntity>): Completable {
        return Completable.fromAction { UserDatabase.get().notificationTrayItemDao().save(items) }
    }

    fun getNotificationTrayItemPagingSource(): PagingSource<Int, NotificationTrayItemEntity> {
        return UserDatabase.get().notificationTrayItemPagingDao().queryNotificationTrayItems()
    }

    fun markSeen(notificationId: String, lastSeenAt: DateTime): Completable {
        return Completable.fromAction {
            val currentSeen =
                UserDatabase.get().notificationTrayItemDao().getByIdNow(notificationId)
            if (currentSeen?.lastSeenAt == null || lastSeenAt.isAfter(currentSeen.lastSeenAt)) {
                UserDatabase.get().notificationTrayItemDao().markAsSeen(notificationId, lastSeenAt)
            }
        }
    }
}