package com.amity.socialcloud.sdk.api.social.comment.update

import com.amity.socialcloud.sdk.core.mention.AmityMentionType
import com.amity.socialcloud.sdk.model.core.mention.AmityMentioneeTarget
import com.amity.socialcloud.sdk.model.social.comment.AmityComment
import com.google.gson.JsonArray
import com.google.gson.JsonObject

class AmityAttachmentCommentUpdate internal constructor(
    commentId: String,
    private val text: String?,
    private val attachments: List<AmityComment.Attachment>?,
    metadata: JsonObject?,
    mentionees: List<AmityMentioneeTarget>?,
) : AmityCommentUpdate(commentId, metadata, mentionees) {

    override fun getData(): JsonObject? {
        if (text == null) return null

        val data = JsonObject()
        data.addProperty("text", text)
        return data
    }

    override fun getAttachments(): JsonArray? {
        if (attachments == null) return null

        val data = JsonArray()
        attachments.forEach { attachment ->
            with(JsonObject()) {
                when (attachment) {
                    is AmityComment.Attachment.IMAGE -> {
                        addProperty("type", attachment.getDataType().apiKey)
                        addProperty("fileId", attachment.getFileId())
                        data.add(this)
                    }
                }
            }
        }
        return data
    }

    class Builder internal constructor(private val commentId: String) {

        private var text: String? = null
        private var attachments: List<AmityComment.Attachment>? = null
        private var metadata: JsonObject? = null
        private var mentionees: List<AmityMentioneeTarget>? = null

        /**
         * @param [metadata] to replace metadata of the comment
         * @return the same instance of AmityAttachmentCommentUpdate.Builder
         */
        fun metadata(metadata: JsonObject): Builder {
            this.metadata = metadata
            return this
        }

        /**
         * @param [userIds] to update the list of userId of users to be mentioned by this comment
         * @return the same instance of AmityAttachmentCommentUpdate.Builder
         */
        fun mentionUsers(userIds: List<String>): Builder {
            this.mentionees = listOf(AmityMentioneeTarget(AmityMentionType.USER.apiKey, userIds))
            return this
        }

        /**
         * @param [text] to replace text value of the comment
         * @return the same instance of AmityAttachmentCommentUpdate.Builder
         */
        fun text(text: String): Builder {
            return apply {
                this.text = text
            }
        }

        /**
         * @param [attachments] to replace attachments of the comment
         * @return the same instance of AmityAttachmentCommentUpdate.Builder
         */
        fun attachments(vararg commentAttachments: AmityComment.Attachment): Builder {
            return apply {
                this.attachments = commentAttachments.toList()
            }
        }

        /**
         * Instantiates AmityAttachmentCommentUpdate with built params.
         * @return new instance of AmityAttachmentCommentUpdate.
         */
        fun build(): AmityAttachmentCommentUpdate {
            return AmityAttachmentCommentUpdate(
                commentId,
                text,
                attachments,
                metadata,
                mentionees
            )
        }
    }

}