package com.amity.socialcloud.sdk.core

import com.amity.socialcloud.sdk.chat.data.channel.ChannelRepository
import com.amity.socialcloud.sdk.chat.data.marker.subchannel.SubChannelMarkerRepository
import com.amity.socialcloud.sdk.core.session.component.SessionComponent
import com.amity.socialcloud.sdk.core.session.eventbus.NetworkConnectionEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.core.session.model.NetworkConnectionEvent
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.amity.socialcloud.sdk.log.AmityLog
import com.amity.socialcloud.sdk.social.data.post.PostRepository
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.schedulers.Schedulers
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.TimeUnit

internal class ObjectResolverEngine(
	sessionLifeCycleEventBus: SessionLifeCycleEventBus,
	sessionStateEventBus: SessionStateEventBus
) : SessionComponent(sessionLifeCycleEventBus, sessionStateEventBus) {
	
	private var isActive: Boolean = false
	private var isOnline: Boolean = true
	private val buffer = ConcurrentHashMap<ReferenceType, MutableList<String>>()
	private var isResolvingTask: Boolean = false
	
	// Timer every 5 seconds to ensure that no sync job is missed
	private val timer = Flowable.interval(TIMER_INTERVAL, TimeUnit.MILLISECONDS)
			.filter { buffer.isNotEmpty() }
			.flatMapCompletable {
				resolveObjects()
			}
			.onErrorComplete()
			.subscribeOn(Schedulers.io())
	
	init {
		NetworkConnectionEventBus
			.observe()
			.doOnNext { event  ->
				when (event) {
					is NetworkConnectionEvent.Connected -> {
						isOnline = true
					}
					else -> {
						isOnline = false
					}
				}
			}
				.subscribeOn(Schedulers.io())
				.subscribe()
		
		timer.subscribe()
	}
	
	private fun resolveObjects(): Completable {
		return if (!isResolvingTask && isActive && isOnline) {
			isResolvingTask = true
			Flowable.fromIterable(buffer.keys)
				.flatMapCompletable(::fetchByIds)
				.doOnError {
					AmityLog.e("Fail to resolve ids: ${buffer.values.toList()}")
				}
		} else {
			Completable.complete()
		}
	}
	
	fun resolve(id: String, referenceType: ReferenceType) {
		val objectIdList = buffer.get(referenceType) ?: mutableListOf()
		if (!objectIdList.contains(id)) {
			if (objectIdList.size >= BUFFER_ID_LIMIT) {
				objectIdList.removeFirst() // Remove first element of the array
			}
			objectIdList.add(id)
			buffer[referenceType] = objectIdList
		}
	}
	
	fun resolve(ids: List<String>, referenceType: ReferenceType) {
		ids.map {
			resolve(it, referenceType)
		}
	}
	
	private fun fetchByIds(type: ReferenceType): Completable {
		val ids = buffer.get(type)?.distinct()
		return if (ids?.isNotEmpty() == true) {
			when(type) {
				ReferenceType.CHANNEL -> ids.let(ChannelRepository()::getChannels).ignoreElements()
				ReferenceType.POST -> ids.let(PostRepository()::getPostByIds).ignoreElements()
				ReferenceType.SUBCHANNEL_UNREAD_INFO -> ids.let(SubChannelMarkerRepository()::fetchSubChannelMarker)
				ReferenceType.CHANNEL_UNREAD_INFO -> ids.let(SubChannelMarkerRepository()::fetchSubChannelUnreadInfo)
			}
				.doOnSubscribe {
					AmityLog.e("Resolving $type: $ids")
					// Clear buffer
					clearBuffer(type)
					isResolvingTask = false
				}
				.onErrorComplete()
		} else {
			Completable.complete()
		}
	}
	
	private fun clearBuffer(type: ReferenceType? = null) {
		type
			?.let(buffer::remove)
			?: buffer.clear()
	}
	
	override fun onSessionStateChange(sessionState: SessionState) {
		isActive = when(sessionState) {
			SessionState.Established -> {
				true
			}
			else -> {
				false
			}
		}
	}
	
	override fun establish(account: EkoAccount) {
		isActive = true
	}
	
	override fun destroy() {
		isActive = false
		clearBuffer()
		isResolvingTask = false
	}
	
	override fun handleTokenExpire() {
		isActive = false
	}
	
	companion object {
		enum class ReferenceType {
			POST,
			CHANNEL,
			SUBCHANNEL_UNREAD_INFO,
			CHANNEL_UNREAD_INFO,
		}
		
		private const val  TIMER_INTERVAL = 1000L
		private const val  BUFFER_ID_LIMIT = 100
		
	}

}