package com.amity.socialcloud.sdk.core

import com.amity.socialcloud.sdk.chat.domain.marker.message.SyncMarkedMessageUseCase
import com.amity.socialcloud.sdk.chat.domain.marker.reader.*
import com.amity.socialcloud.sdk.core.session.component.SessionComponent
import com.amity.socialcloud.sdk.core.session.eventbus.MarkerEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.ekoapp.ekosdk.internal.api.dto.MarkedMessageDto
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.disposables.CompositeDisposable
import io.reactivex.rxjava3.schedulers.Schedulers
import java.util.concurrent.TimeUnit

internal class MarkReadEngine(
	sessionLifeCycleEventBus: SessionLifeCycleEventBus,
	sessionStateEventBus: SessionStateEventBus
) : SessionComponent(sessionLifeCycleEventBus, sessionStateEventBus) {
	
	// Map contains subChannelId and isReading flag
	// to determine which subChannel is on reading state
	private val isReadingMap = hashMapOf<String, Boolean>()
	
	private val disposable = CompositeDisposable()
	
	private val timer = Flowable.interval(SYNC_INTERVAL,TimeUnit.SECONDS)
			.flatMapCompletable {
				timerDidTrigger()
			}
			.subscribeOn(Schedulers.io())
	
	private var isReadyToRead: Boolean = false
	
	init {
		MarkerEventBus
			.observe()
			.filter { it is MarkerEvent.MarkedMessage || it is MarkerEvent.NetworkConnection.Connected}
			.flatMapCompletable { event  ->
				when (event) {
					is MarkerEvent.MarkedMessage -> {
						syncReadAndDeliveredCount(event.dto)
					}
					is MarkerEvent.NetworkConnection.Connected -> {
						// Start reading the reading subChannel again
						// when resume from offline state
						resumeReading().onErrorComplete()
					}
					else -> {
						Completable.complete()
					}
				}
			}
				.subscribeOn(Schedulers.io())
				.subscribe()
	}
	
	private fun timerDidTrigger(): Completable {
		return readSubChannels()
				.onErrorComplete()
	}
	
	fun startReading(subChannelId: String, channelId: String? = null): Completable {
		return Single.just(subChannelId)
			.flatMapCompletable {
				isReadingMap[subChannelId] = true
				StartSubChannelReadingUseCase().execute(subChannelId)
					.doOnError { isReadingMap[subChannelId] = false }
			}.andThen(
				SubscribeToSubChannelMarkerTopicUseCase().execute(subChannelId, channelId)
			)
	}
	
	private fun readSubChannels(): Completable {
		val readingSubChannelIds = getReadingSubChannelId()
		return when {
			readingSubChannelIds.isEmpty() -> Completable.complete()
			else -> {
				if (isReadyToRead) {
					ReadSubChannelUseCase().execute(readingSubChannelIds)
				} else {
					Completable.complete()
				}
			}
		}
	}
	
	fun stopReading(subChannelId: String, channelId: String? = null): Completable {
		isReadingMap[subChannelId] = false
		return StopSubChannelReadingUseCase().execute(subChannelId)
			.andThen(
				UnsubscribeToSubChannelMarkerTopicUseCase().execute(subChannelId, channelId)
			)
	}
	
	private fun getReadingSubChannelId(): List<String> {
		return isReadingMap.filter { it.value }.keys.toList()
	}
	
	private fun resumeReading(): Completable {
		return readSubChannels()
	}
	
	private fun syncReadAndDeliveredCount(dto: MarkedMessageDto): Completable {
		return SyncMarkedMessageUseCase().execute(dto).onErrorComplete()
	}
	
	private fun startTimer() {
		if (disposable.size() == 0 || disposable.isDisposed){
			disposable.clear()
			timer.subscribe().let(disposable::add)
		}
	}
	
	private fun stopTimer() {
		if (disposable.size() > 0){
			disposable.clear()
		}
	}
	
	override fun onSessionStateChange(sessionState: SessionState) {
		when(sessionState) {
			SessionState.Established -> {
				isReadyToRead = true
			}
			else -> {
				isReadyToRead = false
			}
		}
	}
	
	override fun establish(account: EkoAccount) {
		isReadyToRead = true
		startTimer()
	}
	
	override fun destroy() {
		isReadyToRead = false
		stopTimer()
		isReadingMap.clear()
	}
	
	override fun handleTokenExpire() {
		isReadyToRead = false
		stopTimer()
	}
	
	companion object {
		private const val SYNC_INTERVAL = 30L
	}
}