package com.amity.socialcloud.sdk.model.core.notificationtray

import android.os.Parcelable
import com.amity.socialcloud.sdk.core.JsonObjectParceler
import com.amity.socialcloud.sdk.core.domain.notificationtray.seen.MarkNotificationItemSeenUseCase
import com.amity.socialcloud.sdk.model.core.user.AmityUser
import com.ekoapp.ekosdk.ReactorObject
import com.google.common.base.Objects
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import kotlinx.parcelize.Parcelize
import kotlinx.parcelize.TypeParceler
import org.joda.time.DateTime

@Parcelize
@TypeParceler<JsonObject?, JsonObjectParceler>
data class AmityNotificationTrayItem internal constructor(
    var targetId: String,
    var targetType: String,
    var actionType: String,
    var actionReferenceId: String,
    var actors: AmityNotificationTrayActorIdMap = AmityNotificationTrayActorIdMap(),
    var actorsCount: Int,
    var data: Data?,
    var id: String,
    var daySegment: DateTime?,
    var lastOccurredAt: DateTime?,
    var lastSeenAt: DateTime?,
    var parentId: String,
    var referenceId: String,
    var referenceType: String,
    var templatedText: String,
    var text: String,
    var trayItemCategory: String,
    var isSeen: Boolean? = null,
    var isRecent: Boolean? = null,
    internal var actorUsers: List<AmityUser>? = null,
): Parcelable, ReactorObject {

    sealed class Data : Parcelable {

        @Parcelize
        @TypeParceler<JsonObject?, JsonObjectParceler>
        class COMMUNITY(
            internal val name: String,
            internal val rawData: JsonObject?,
        ) : Data() {

            fun getText(): String {
                return name
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is COMMUNITY
                        && Objects.equal(other.name, name))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(name, rawData)
            }

        }

        @Parcelize
        @TypeParceler<JsonObject?, JsonObjectParceler>
        class REACTION(
            internal val names: List<String>,
            internal val rawData: JsonObject?,
        ) : Data() {

            fun getText(): List<String> {
                return names
            }

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is REACTION
                        && Objects.equal(other.names, names))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(names, rawData)
            }

        }

        @Parcelize
        @TypeParceler<JsonObject?, JsonObjectParceler>
        class CUSTOM(
            internal val rawData: JsonObject?
        ) : Data() {

            override fun equals(other: Any?): Boolean {
                return (other != null
                        && other is CUSTOM
                        && Objects.equal(other.rawData, rawData))
            }

            override fun hashCode(): Int {
                return Objects.hashCode(rawData)
            }

        }
    }

    sealed class DataType : Parcelable {
        abstract fun getApiKey(): String

        @Parcelize
        object COMMUNITY : DataType() {
            override fun getApiKey(): String = "communityName"
        }

        @Parcelize
        object REACTION : DataType() {
            override fun getApiKey(): String = "reactionName"
        }

        @Parcelize
        class CUSTOM(val customTypeName: String) : DataType() {
            override fun getApiKey(): String = customTypeName
        }

        companion object {
            fun sealedOf(dataType: JsonObject?): DataType {
                if (dataType == null) return CUSTOM("")

                return when {
                    dataType.has(COMMUNITY.getApiKey()) -> COMMUNITY
                    dataType.has(REACTION.getApiKey()) -> REACTION
                    else -> CUSTOM("")
                }
            }
        }
    }

    fun markSeen(): Completable {
        return MarkNotificationItemSeenUseCase().execute(id)
    }

    override fun updatedAt(): DateTime? {
        return lastOccurredAt
    }

    override fun uniqueId(): String {
        return id
    }

    companion object {
        internal fun generateUniqueId(targetType: String, targetId: String): String {
            return "$targetType/$targetId"
        }
    }
}