package com.ekoapp.ekosdk.internal.data;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.room.Room;
import androidx.room.RoomDatabase;
import androidx.room.migration.Migration;

import net.zetetic.database.sqlcipher.SupportOpenHelperFactory;

public final class RoomUtil {

    /**
     * Creates a database builder with fallbackToDestructiveMigration enabled.
     * This is the default behavior for backward compatibility.
     */
    public static <T extends RoomDatabase> RoomDatabase.Builder<T> databaseBuilder(
            @NonNull Context context,
            @NonNull Class<T> klass,
            @NonNull String name,
            @Nullable byte[] passphrase
    ) {
        return databaseBuilder(context, klass, name, passphrase, new Migration[0], -1);
    }

    /**
     * Creates a database builder with migrations support.
     * If migrations array is empty, fallbackToDestructiveMigration is used.
     * If migrations are provided, they will be used instead of destructive fallback.
     */
    public static <T extends RoomDatabase> RoomDatabase.Builder<T> databaseBuilder(
            @NonNull Context context,
            @NonNull Class<T> klass,
            @NonNull String name,
            @Nullable byte[] passphrase,
            @NonNull Migration[] migrations
    ) {
        return databaseBuilder(context, klass, name, passphrase, migrations, -1);
    }

    /**
     * Creates a database builder with migrations support and fallback from specific versions.
     *
     * @param migrations Array of migrations to apply
     * @param fallbackFromVersion If > 0, versions before this will use destructive migration.
     *                           If <= 0, and migrations is empty, fallbackToDestructiveMigration is used.
     */
    public static <T extends RoomDatabase> RoomDatabase.Builder<T> databaseBuilder(
            @NonNull Context context,
            @NonNull Class<T> klass,
            @NonNull String name,
            @Nullable byte[] passphrase,
            @NonNull Migration[] migrations,
            int fallbackFromVersion
    ) {
        RoomDatabase.Builder<T> builder = Room.databaseBuilder(context, klass, name);

        if (migrations.length > 0) {
            // Use migrations
            builder.addMigrations(migrations);

            // If fallbackFromVersion specified, allow destructive migration for older versions
            if (fallbackFromVersion > 0) {
                // Create array of versions to fallback from (1 to fallbackFromVersion-1)
                int[] versionsToFallback = new int[fallbackFromVersion - 1];
                for (int i = 0; i < versionsToFallback.length; i++) {
                    versionsToFallback[i] = i + 1;
                }
                if (versionsToFallback.length > 0) {
                    builder.fallbackToDestructiveMigrationFrom(versionsToFallback);
                }
            }
        } else {
            // No migrations - fallback to destructive for all
            builder.fallbackToDestructiveMigration();
        }

        if (passphrase != null) {
            builder.openHelperFactory(new SupportOpenHelperFactory(passphrase, null, false));
        }
        return builder;
    }

    private RoomUtil() { /* NO INSTANCE */ }
}
