package com.ekoapp.ekosdk.internal.data;

import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.room.RoomDatabase;
import androidx.room.migration.Migration;

abstract class BaseRoomDatabase extends RoomDatabase {

    private static final Migration[] NO_MIGRATIONS = new Migration[0];

    /**
     * Initialize database without migrations (uses fallbackToDestructiveMigration).
     */
    static synchronized <T extends BaseRoomDatabase> T init(
            @NonNull Context context,
            @NonNull Class<T> klass,
            @NonNull String name,
            @Nullable T database,
            boolean allowMainThreadQueries,
            @Nullable byte[] passphrase) {
        return init(context, klass, name, database, allowMainThreadQueries, passphrase, NO_MIGRATIONS, -1);
    }

    /**
     * Initialize database with migrations support.
     * If migrations array is empty, fallbackToDestructiveMigration is used.
     */
    static synchronized <T extends BaseRoomDatabase> T init(
            @NonNull Context context,
            @NonNull Class<T> klass,
            @NonNull String name,
            @Nullable T database,
            boolean allowMainThreadQueries,
            @Nullable byte[] passphrase,
            @NonNull Migration[] migrations) {
        return init(context, klass, name, database, allowMainThreadQueries, passphrase, migrations, -1);
    }

    /**
     * Initialize database with migrations support and fallback for old versions.
     *
     * @param migrations Array of migrations to apply
     * @param fallbackFromVersion Versions before this will use destructive migration
     */
    static synchronized <T extends BaseRoomDatabase> T init(
            @NonNull Context context,
            @NonNull Class<T> klass,
            @NonNull String name,
            @Nullable T database,
            boolean allowMainThreadQueries,
            @Nullable byte[] passphrase,
            @NonNull Migration[] migrations,
            int fallbackFromVersion) {
        Context appContext = context.getApplicationContext();
        if (database != null) {
            database.close();
        }

        final RoomDatabase.Builder<T> builder = RoomUtil.databaseBuilder(
                appContext, klass, name, passphrase, migrations, fallbackFromVersion);
        builder.setJournalMode(JournalMode.AUTOMATIC);
        if (allowMainThreadQueries) {
            builder.allowMainThreadQueries();
        }
        return builder.build();
    }
}
