package com.amity.socialcloud.sdk.social.data.story

import androidx.paging.PagingSource
import com.amity.socialcloud.sdk.api.social.story.AmityStorySortOption
import com.amity.socialcloud.sdk.core.data.session.SessionLocalDataStore
import com.amity.socialcloud.sdk.entity.social.story.StoryEntity
import com.amity.socialcloud.sdk.model.social.story.AmityStory
import com.amity.socialcloud.sdk.model.social.story.AmityStoryImageDisplayMode
import com.amity.socialcloud.sdk.model.social.story.AmityStoryItem
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.google.gson.JsonArray
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers
import org.joda.time.DateTime


internal class StoryLocalDataStore {

    fun saveStories(stories: List<StoryEntity>): Completable {
        return Completable.fromAction {
            val refStories = stories.filter { it.storyId != it.uniqueId }
            val noRefStories = stories.filter { it.storyId == it.uniqueId }
            val savingStories = mutableListOf<StoryEntity>()
            for (story in noRefStories) {
                val cachedObject =
                    UserDatabase.get().storyDao().getByStoryIdNow(story.storyId)
                if (cachedObject != null) {
                    story.uniqueId = cachedObject.uniqueId
                }
                savingStories.add(story)
            }
            for (story in refStories) {
                val cachedObject =
                    UserDatabase.get().storyDao().getByIdNow(story.uniqueId)
                if (cachedObject != null) {
                    if(cachedObject.uniqueId == cachedObject.storyId
                            && story.updatedAt?.isBefore(cachedObject.updatedAt) ?: true) {
                        story.createdAt = cachedObject.createdAt
                        story.updatedAt = cachedObject.updatedAt
                    }
                } else {
                    story.uniqueId = story.storyId
                }
                savingStories.add(story)
            }
            UserDatabase.get().storyDao().save(savingStories)
        }.subscribeOn(Schedulers.io())
    }

    fun createImageStory(
        referenceId: String,
        targetType: AmityStory.TargetType,
        targetId: String,
        storyItems: List<AmityStoryItem> = emptyList(),
        metadata: JsonObject?,
        imageDisplayMode: AmityStoryImageDisplayMode
    ): Completable {
        return Completable.fromAction {
            val now = DateTime.now()
            val expiresAt = now.let {
                it.plusDays(365)
            }
            val story = StoryEntity().apply {
                this.uniqueId = referenceId
                this.storyId = referenceId
                this.targetType = targetType.apiKey
                this.targetId = targetId
                this.dataType = AmityStory.DataType.IMAGE.apiKey
                this.syncState = AmityStory.State.SYNCING.apiKey
                this.data = JsonObject().apply {
                    addProperty("imageDisplayMode", imageDisplayMode.apiKey)
                    addProperty("fileId", referenceId)
                }
                this.items = JsonArray().apply {
                    storyItems.forEach {
                        add(it.getPayload())
                    }
                }
                this.metadata = metadata
                this.createdAt = now
                this.updatedAt = now
                this.storyExpiresAt = expiresAt
                this.creatorId = SessionLocalDataStore().getActiveUserId()

            }
            UserDatabase.get().storyDao().save(listOf(story))
        }
    }

    fun createVideoStory(
        referenceId: String,
        targetType: AmityStory.TargetType,
        targetId: String,
        storyItems: List<AmityStoryItem> = emptyList(),
        metadata: JsonObject?,
    ): Completable {
        return Completable.fromAction {
            val now = DateTime.now()
            val expiresAt = now.let {
                it.plusDays(365)
            }
            val story = StoryEntity().apply {
                this.uniqueId = referenceId
                this.storyId = referenceId
                this.targetType = targetType.apiKey
                this.targetId = targetId
                this.syncState = AmityStory.State.SYNCING.apiKey
                this.dataType = AmityStory.DataType.VIDEO.apiKey
                this.data = JsonObject().apply {
                    val original = JsonObject().apply {
                        addProperty("original", referenceId)
                    }
                    add("videoFileId", original)
                }
                this.items = JsonArray().apply {
                    storyItems.forEach {
                        add(it.getPayload())
                    }
                }
                this.metadata = metadata
                this.createdAt = now
                this.updatedAt = now
                this.storyExpiresAt = expiresAt
                this.creatorId = SessionLocalDataStore().getActiveUserId()
            }
            UserDatabase.get().storyDao().save(listOf(story))
        }
    }

    fun updateStorySyncState(
        referenceId: String,
        syncState: AmityStory.State
    ): Completable {
        return UserDatabase.get().storyDao().updateSyncState(referenceId, syncState.apiKey)
    }

    fun getStoryPagingSource(
        targetType: AmityStory.TargetType,
        targetId: String,
        sortOption: AmityStorySortOption
    ): PagingSource<Int, StoryEntity> {
        return UserDatabase.get().storyPagingDao().getStoriesPagingSource(
            targetType,
            targetId,
            sortOption
        )
    }

    fun getLatestStories(
        targetType: AmityStory.TargetType,
        targetId: String,
        hash: Int,
        nonce: Int
    ): Flowable<StoryEntity> {
        return UserDatabase.get().storyDao()
            .getLatestStory(
                targetType,
                targetId,
                hash,
                nonce
            )
    }

    fun observeStory(storyId: String): Flowable<StoryEntity> {
        return UserDatabase.get().storyDao().observeStory(storyId = storyId)
    }

    fun getStoryUniqueId(storyId: String): String? {
        return UserDatabase.get().storyDao().getUniqueIdByStoryId(storyId)
    }

    fun deleteStory(storyId: String): Completable {
        return UserDatabase.get().storyDao().hardDeleteStory(storyId)
    }

    fun getStoryNow(storyId: String): StoryEntity? {
        return UserDatabase.get().storyDao().getByStoryIdNow(storyId)
    }

    fun getHighestStoryExpiresAt(
        targetType: AmityStory.TargetType,
        targetId: String,
        syncStates: List<AmityStory.State>
    ): DateTime? {
        return UserDatabase.get().storyDao().getHighestStoryExpiresAt(
            targetType = targetType.apiKey,
            targetId = targetId,
            syncStates = syncStates.map { it.apiKey }
        )
    }

    fun getStoryCount(
        targetType: AmityStory.TargetType,
        targetId: String,
        syncStates: List<AmityStory.State>
    ): Int {
        return UserDatabase.get().storyDao().getStoryCount(
            targetType.apiKey,
            targetId,
            syncStates.map { it.apiKey }
        )
    }

    fun triggerStoryReload(storyId: String) : Completable{
        return UserDatabase.get().storyDao().dummyUpdateStory(storyId)
    }

    fun findCache(targetType: AmityStory.TargetType, targetId: String): Single<List<String>> {
        return Single.fromCallable {
            UserDatabase.get().storyDao().findCache(targetType.apiKey, targetId).map { it.uniqueId }
        }
    }

    fun decrementCommentCount(storyId: String) {
        UserDatabase.get().storyDao().decrementCommentCount(storyId)
    }

    fun hasInLocal(storyId: String): Boolean {
        return UserDatabase.get().storyDao().getByStoryIdNow(storyId) != null
    }

    fun getStoriesByTargets(targets: List<Pair<AmityStory.TargetType, String>>): Flowable<List<StoryEntity>> {
        return UserDatabase.get().storySinglePageDao().getStoryTargetByTargets(targets)
    }

}