package com.amity.socialcloud.sdk.social.data.post.paging

import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.amity.socialcloud.sdk.social.data.post.PostRepositoryHelper
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator

internal class UserPostKeyCreator(
    private val userId: String,
    private val sortBy: String,
    private val isDeleted: Boolean?,
    private val dataTypes: List<AmityPost.DataType>,
    private val tags: List<String>,
    private val includeMixedStructure: Boolean
) : DynamicQueryStreamKeyCreator {

    override fun toMap(): Map<String, Any> {
        return mapOf(
            "targetId" to userId,
            "targetType" to AmityPost.TargetType.USER.apiKey,
            "sortBy" to sortBy,
            "isDeleted" to (isDeleted ?: false),
            "dataTypes" to PostRepositoryHelper.convertPostTypesToString(dataTypes),
            "tags" to tags.joinToString(","),
            "includeMixedStructure" to includeMixedStructure,
        )
    }

    override fun getFilterQuery(): String {
        return getConditionStatement(
            userId = userId,
            includeDeleted = isDeleted,
            dataTypes = dataTypes,
            tags = tags,
            includeMixedStructure = includeMixedStructure
        )
    }

    private fun getConditionStatement(
        userId: String,
        includeDeleted: Boolean?,
        dataTypes: List<AmityPost.DataType>,
        tags: List<String>,
        includeMixedStructure: Boolean
    ): String {
        //userId filter
        val userFilterStatement = "post.targetType = '${AmityPost.TargetType.USER.apiKey}'" +
                " and post.targetId = '$userId'"

        //includeDelete filter
        var isDeletedFilterStatement = ""
        if (includeDeleted == false) {
            isDeletedFilterStatement = " and post.isDeleted = 0"
        } else if (includeDeleted == true) {
            isDeletedFilterStatement = " and post.isDeleted = 1"
        }
        //post types filter
        var postTypesFilterStatement = ""
        if (dataTypes.isNotEmpty()) {
            postTypesFilterStatement =
                " and post.postDataType IN (${getPostTypesAsArrayString(dataTypes)})"
        }
        //tags filter
        var tagsFilterStatement = ""
        if (tags.isNotEmpty()) {
            tagsFilterStatement = " and " + tags.joinToString(" and ") { tag ->
                "post.tags LIKE '%\"$tag\"%'"
            }
        }
        //structureType filter - exclude mixed structure posts when includeMixedStructure is false
        var structureTypeFilterStatement = ""
        if (!includeMixedStructure) {
            structureTypeFilterStatement = " and (post.structureType != 'mixed' OR post.structureType IS NULL)"
        }
        //parentPost filter
        val parentPostFilter = if (dataTypes.isNullOrEmpty()) {
            " and post.parentPostId is null"
        } else {
            " and post.targetId is not null"
        }

        //compose all condition
        return userFilterStatement +
                isDeletedFilterStatement +
                postTypesFilterStatement +
                tagsFilterStatement +
                structureTypeFilterStatement +
                parentPostFilter
    }

    fun getPostTypesAsArrayString(dataTypes: List<AmityPost.DataType>): String {
        val typesString = StringBuilder()
        dataTypes.forEachIndexed { index, dataType ->
            if (index == 0) {
                typesString.append("'${dataType.getApiKey()}'")
            } else {
                typesString.append(", '${dataType.getApiKey()}'")
            }
        }
        return typesString.toString()
    }

}