package com.amity.socialcloud.sdk.social.data.post.paging

import com.amity.socialcloud.sdk.model.social.feed.AmityFeedType
import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.amity.socialcloud.sdk.social.data.post.PostRepositoryHelper
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator

internal class CommunityPostKeyCreator(
    private val communityId: String,
    private val sortBy: String,
    private val isDeleted: Boolean?,
    private val feedType: AmityFeedType,
    private val dataTypes: List<AmityPost.DataType>,
    private val tags: List<String>,
    private val matchingOnlyParentPosts: Boolean,
    private val includeMixedStructure: Boolean
) : DynamicQueryStreamKeyCreator {

    override fun toMap(): Map<String, Any> {
        return mapOf(
            "targetId" to communityId,
            "targetType" to AmityPost.TargetType.COMMUNITY.apiKey,
            "sortBy" to sortBy,
            "isDeleted" to (isDeleted ?: false),
            "feedType" to feedType.apiKey,
            "dataTypes" to PostRepositoryHelper.convertPostTypesToString(dataTypes),
            "tags" to tags.joinToString(","),
            "matchingOnlyParentPosts" to matchingOnlyParentPosts,
            "includeMixedStructure" to includeMixedStructure,
        )
    }

    override fun getFilterQuery(): String {
        return getConditionStatement(
            communityId = communityId,
            feedType = feedType.apiKey,
            includeDeleted = isDeleted,
            dataTypes = dataTypes,
            tags = tags,
            matchingOnlyParentPosts = matchingOnlyParentPosts,
            includeMixedStructure = includeMixedStructure
        )
    }

    private fun getConditionStatement(
        communityId: String,
        feedType: String,
        includeDeleted: Boolean?,
        dataTypes: List<AmityPost.DataType>,
        tags: List<String>,
        matchingOnlyParentPosts: Boolean,
        includeMixedStructure: Boolean
    ): String {
        //communityId filter
        val communityFilterStatement =
            "post.targetType = '${AmityPost.TargetType.COMMUNITY.apiKey}'" +
                    " and post.targetId = '$communityId'"

        //feedType filter
        val feedTypeFilterStatement = " and post.feedType = '$feedType'"

        //includeDelete filter
        var isDeletedFilterStatement = ""
        if (includeDeleted == false) {
            isDeletedFilterStatement = " and post.isDeleted = 0"
        } else if (includeDeleted == true) {
            isDeletedFilterStatement = " and post.isDeleted = 1"
        }
        //post types filter
        var postTypesFilterStatement = ""
        if (dataTypes.isNotEmpty()) {
            postTypesFilterStatement =
                " and post.postDataType IN (${getPostTypesAsArrayString(dataTypes)})"
        }
        //tags filter
        var tagsFilterStatement = ""
        if (tags.isNotEmpty()) {
            tagsFilterStatement = " and " + tags.joinToString(" and ") { tag ->
                "post.tags LIKE '%\"$tag\"%'"
            }
        }
        //structureType filter - exclude mixed structure posts when includeMixedStructure is false
        var structureTypeFilterStatement = ""
        if (!includeMixedStructure) {
            structureTypeFilterStatement = " and (post.structureType != 'mixed' OR post.structureType IS NULL)"
        }
        //parentPost filter
        val parentPostFilter = if (matchingOnlyParentPosts) {
            " and post.parentPostId is null"
        } else {
            " and post.targetId is not null"
        }

        //compose all condition
        return communityFilterStatement +
                feedTypeFilterStatement +
                isDeletedFilterStatement +
                postTypesFilterStatement +
                tagsFilterStatement +
                structureTypeFilterStatement +
                parentPostFilter
    }

    private fun getPostTypesAsArrayString(postTypes: List<AmityPost.DataType>): String {
        val typesString = StringBuilder()
        postTypes.forEachIndexed { index, dataType ->
            if (index == 0) {
                typesString.append("'${dataType.getApiKey()}'")
            } else {
                typesString.append(", '${dataType.getApiKey()}'")
            }
        }
        return typesString.toString()
    }
}