package com.amity.socialcloud.sdk.social.data.post.engine

import com.ekoapp.ekosdk.internal.data.UserDatabase
import org.joda.time.DateTime
import java.util.concurrent.ConcurrentLinkedQueue

/**
 * Singleton engine for managing local comment count computation.
 * This engine processes comment change tasks and updates the local comment count
 * for posts in a batched manner to prevent blocking the main thread.
 */
internal object PostCommentCountEngine {

    private const val BATCH_SIZE = 50

    private val taskQueue = ConcurrentLinkedQueue<CommentChangeTask>()
    private val commentChangeTrackers = mutableMapOf<String, CommentChangeTracker>()
    private var isProcessing = false

    /**
     * Queue a comment change task and trigger processing.
     */
    @Synchronized
    fun queueCommentChangeTask(task: CommentChangeTask) {
        taskQueue.add(task)
        processCommentChangeTasks()
    }

    /**
     * Process tasks in capped batches, recurse until queue is empty.
     */
    @Synchronized
    private fun processCommentChangeTasks() {
        if (isProcessing) return
        isProcessing = true

        var processedCount = 0
        val postsToUpdate = mutableSetOf<String>()

        while (processedCount < BATCH_SIZE && taskQueue.isNotEmpty()) {
            val task = taskQueue.poll() ?: break
            val postId = task.postId

            when (task) {
                is ResetTask -> {
                    if (processResetTaskInternal(task)) {
                        postsToUpdate.add(postId)
                    }
                }
                is CreateTask -> {
                    if (processCreateTaskInternal(task)) {
                        postsToUpdate.add(postId)
                    }
                }
                is DeleteTask -> {
                    if (processDeleteTaskInternal(task)) {
                        postsToUpdate.add(postId)
                    }
                }
            }
            processedCount++
        }

        // Publish updates for all affected posts
        postsToUpdate.forEach { postId ->
            publishUpdate(postId)
        }

        isProcessing = false

        // Continue processing if there are more tasks
        if (taskQueue.isNotEmpty()) {
            processCommentChangeTasks()
        }
    }

    /**
     * Process a reset task - resets the tracker state for a post.
     * @return true if the tracker was updated
     */
    private fun processResetTaskInternal(task: ResetTask): Boolean {
        val tracker = CommentChangeTracker(
            latestCommentCreatedAt = task.latestCreatedAt,
            serverCommentCount = task.serverCommentCount
        )
        commentChangeTrackers[task.postId] = tracker
        return true
    }

    /**
     * Process a create task - tracks a comment creation.
     * @return true if the comment was added (not a duplicate or stale)
     */
    private fun processCreateTaskInternal(task: CreateTask): Boolean {
        // Skip if no tracker exists - no baseline data means no one is observing this post
        val tracker = commentChangeTrackers[task.postId] ?: return false

        // Skip if this is a stale event (comment created before the latest known comment)
        if (task.createdAt.isBefore(tracker.latestCommentCreatedAt)) {
            return false
        }

        // Skip if already tracked
        if (tracker.createdCommentIds.contains(task.commentId)) {
            return false
        }

        // Remove from deleted set if it was previously marked as deleted
        tracker.deletedCommentIds.remove(task.commentId)

        // Add to created set
        tracker.createdCommentIds.add(task.commentId)
        return true
    }

    /**
     * Process a delete task - tracks a comment deletion.
     * @return true if the comment was added to deleted set
     */
    private fun processDeleteTaskInternal(task: DeleteTask): Boolean {
        // Skip if no tracker exists - no baseline data means no one is observing this post
        val tracker = commentChangeTrackers[task.postId] ?: return false

        // If it was in created set, just remove it (it was a local optimistic add)
        if (tracker.createdCommentIds.remove(task.commentId)) {
            return true
        }

        // Skip if already tracked as deleted
        if (tracker.deletedCommentIds.contains(task.commentId)) {
            return false
        }

        // Add to deleted set
        tracker.deletedCommentIds.add(task.commentId)
        return true
    }

    /**
     * Compute the local comment count for a post.
     * Formula: server + created.size - deleted.size (minimum 0)
     */
    private fun computeCommentCount(postId: String): Int {
        val tracker = commentChangeTrackers[postId] ?: return 0
        val count = tracker.serverCommentCount + 
            tracker.createdCommentIds.size - 
            tracker.deletedCommentIds.size
        return maxOf(0, count)
    }

    /**
     * Publish the update to the Post's localCommentCount in the database.
     */
    private fun publishUpdate(postId: String) {
        val localCommentCount = computeCommentCount(postId)
        UserDatabase.get().postDao().updateLocalCommentCount(postId, localCommentCount)
    }

    /**
     * Clear all tracking data. Useful for testing or logout.
     */
    @Synchronized
    fun clear() {
        taskQueue.clear()
        commentChangeTrackers.clear()
        isProcessing = false
    }

    /**
     * Get the current local comment count for a post (for testing purposes).
     */
    internal fun getLocalCommentCount(postId: String): Int {
        return computeCommentCount(postId)
    }
}

/**
 * Base interface for comment change tasks.
 */
internal sealed interface CommentChangeTask {
    val postId: String
}

/**
 * Task to reset the comment count tracker for a post.
 * This should be queued when receiving post data from the server.
 */
internal data class ResetTask(
    override val postId: String,
    val latestCreatedAt: DateTime,
    val serverCommentCount: Int
) : CommentChangeTask

/**
 * Task to track a comment creation.
 * This should be queued after a successful create comment API response or RTE event.
 */
internal data class CreateTask(
    override val postId: String,
    val commentId: String,
    val createdAt: DateTime
) : CommentChangeTask

/**
 * Task to track a comment deletion.
 * This should be queued after a successful delete comment API response or RTE event.
 */
internal data class DeleteTask(
    override val postId: String,
    val commentId: String
) : CommentChangeTask

/**
 * Internal data structure to track comment changes for a post.
 */
internal data class CommentChangeTracker(
    var latestCommentCreatedAt: DateTime,
    var serverCommentCount: Int,
    val createdCommentIds: MutableSet<String> = mutableSetOf(),
    val deletedCommentIds: MutableSet<String> = mutableSetOf()
)
