package com.amity.socialcloud.sdk.model.core.session

/**
 * Handler interface for automatic access token renewal.
 *
 * This handler is used when the user authenticates via [loginWithAccessToken].
 * The SDK will invoke [onTokenRenew] when the token is about to expire or has expired.
 *
 * Note: This handler is only invoked for sessions created via access token login,
 * not for traditional userId login sessions.
 *
 * Usage:
 * ```kotlin
 * val handler = object : AccessTokenHandler {
 *     override suspend fun onTokenRenew(userId: String): String {
 *         // Fetch new token from your backend
 *         return myBackend.getNewToken(userId)
 *     }
 * }
 * AmityCoreClient.setAccessTokenHandler(handler)
 * AmityCoreClient.loginWithAccessToken(accessToken).submit()
 * ```
 */
interface AccessTokenHandler {

    /**
     * Called when the access token needs to be renewed.
     *
     * This method is invoked when:
     * - The token is about to expire (proactive renewal, ~5 minutes before expiry)
     * - The token has already expired (reactive renewal)
     *
     * @param userId The userId of the current session for which the token is being renewed
     * @return A new JWT access token string
     * @throws Exception if unable to retrieve a new token
     */
    suspend fun onTokenRenew(userId: String): String
}
