package com.amity.socialcloud.sdk.model.core.presence

import com.amity.socialcloud.sdk.core.data.presence.PresenceRepository
import com.amity.socialcloud.sdk.core.domain.user.UserGetByIdsUseCase
import com.amity.socialcloud.sdk.model.core.user.AmityUser
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers
import org.joda.time.DateTime

class AmityOnlineUsersSnapshot internal constructor() {

    private var userPresences: List<AmityUserPresence> = mutableListOf()

    private var users: MutableList<AmityUser> = mutableListOf()
    private var fetchedAt = DateTime.now()
    private var currentPage = 0

    fun getFetchedAt(): DateTime {
        return fetchedAt
    }

    fun getUsers(): List<AmityUser> {
        return users
    }

    fun loadMore(): Completable {
        if (!canLoadMore()) return Completable.complete()

        currentPage += 1
        return loadPage()
    }

    fun canLoadMore(): Boolean {
        return users.size < userPresences.size
    }

    private fun getOnlineUsers(): Completable {
        return PresenceRepository().getOnlineUsers()
            .subscribeOn(Schedulers.io())
            .flatMapCompletable {
                userPresences = it
                fetchedAt = DateTime.now()
                loadPage()
            }
    }

    private fun getOnlineRoomUsers(roomId: String): Completable {
        return PresenceRepository().getRoomUserPresences(roomId = roomId)
            .subscribeOn(Schedulers.io())
            .flatMapCompletable {
                userPresences = it
                fetchedAt = DateTime.now()
                loadPage()
            }
    }

    private fun loadPage(): Completable {
        val paginatedUserPresences = getPaginatedUserPresences()
        return Completable.fromAction {
            paginatedUserPresences
                .map { it.getUserId() }
                .toSet()
                .let { getUsersByIds(it) }
                .let {
                    users.addAll(it)
                }
        }
    }

    private fun getPaginatedUserPresences(): List<AmityUserPresence> {
        if (userPresences.isEmpty()) return emptyList()

        val upperBound = userPresences.size
        val fromIndex = (currentPage * PAGE_SIZE).coerceAtMost(upperBound - 1)
        val toIndex = (fromIndex + PAGE_SIZE).coerceAtMost(upperBound)
        return userPresences.subList(fromIndex, toIndex)
    }

    private fun getUsersByIds(userIds: Set<String>): List<AmityUser> {
        if (userIds.isEmpty()) return emptyList()
        return UserGetByIdsUseCase().execute(userIds)
            .subscribeOn(Schedulers.io())
            .onErrorReturnItem(emptyList())
            .blockingFirst(emptyList())
    }

    companion object {
        private const val PAGE_SIZE = 20

        internal fun create(): Single<AmityOnlineUsersSnapshot> {
            val snapshot = AmityOnlineUsersSnapshot()
            return snapshot.getOnlineUsers()
                .toSingle { snapshot }
        }

        internal fun createByRoomId(roomId: String): Single<AmityOnlineUsersSnapshot> {
            val snapshot = AmityOnlineUsersSnapshot()
            return snapshot.getOnlineRoomUsers(roomId = roomId)
                .toSingle { snapshot }
        }
    }
}