package com.amity.socialcloud.sdk.core.util

import android.util.Base64
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import org.joda.time.DateTime
import org.json.JSONObject

/**
 * Utility class for decoding JWT tokens.
 *
 * This decoder extracts the userId, issuedAt, and expiresAt from a JWT token's payload.
 * It only decodes the payload - it does NOT verify the token signature.
 * Token verification is done server-side via the sessions API.
 */
internal object JwtTokenDecoder {

    private const val JWT_PARTS_COUNT = 3
    private const val PAYLOAD_INDEX = 1

    /**
     * Data class containing decoded JWT token information.
     */
    data class DecodedToken(
        val userId: String,
        val issuedAt: DateTime?,
        val expiresAt: DateTime?
    )

    /**
     * Decodes a JWT token and extracts userId, issuedAt, and expiresAt.
     *
     * Token payload structure (per spec):
     * {
     *   "exp": 1234567890,  // Unix timestamp in seconds
     *   "iat": 1234567890,  // Unix timestamp in seconds
     *   "user": {
     *     "publicUserId": "user123",  // Primary identifier
     *     "userId": "user123"         // Fallback identifier
     *   }
     * }
     *
     * @param token The JWT token string (format: header.payload.signature)
     * @return DecodedToken containing userId, issuedAt, and expiresAt
     * @throws AmityException if the token is malformed or missing required claims
     */
    fun decode(token: String): DecodedToken {
        try {
            val parts = token.split(".")
            if (parts.size != JWT_PARTS_COUNT) {
                throw AmityException.create(
                    "Invalid JWT format: expected 3 parts, got ${parts.size}",
                    null,
                    AmityError.INVALID_PARAMETER
                )
            }

            val payload = decodeBase64(parts[PAYLOAD_INDEX])
            val jsonObject = JSONObject(payload)

            // Extract userId from user.publicUserId or user.userId (per spec)
            val userId = extractUserId(jsonObject)

            // Extract iat (issued at) - Unix timestamp in seconds
            val issuedAt = if (jsonObject.has("iat")) {
                DateTime(jsonObject.getLong("iat") * 1000) // Convert seconds to milliseconds
            } else {
                null
            }

            // Extract exp (expires at) - Unix timestamp in seconds
            val expiresAt = if (jsonObject.has("exp")) {
                DateTime(jsonObject.getLong("exp") * 1000) // Convert seconds to milliseconds
            } else {
                null
            }

            return DecodedToken(
                userId = userId,
                issuedAt = issuedAt,
                expiresAt = expiresAt
            )
        } catch (e: AmityException) {
            throw e
        } catch (e: Exception) {
            throw AmityException.create(
                "Failed to decode JWT token: ${e.message}",
                e,
                AmityError.INVALID_PARAMETER
            )
        }
    }

    /**
     * Extracts the userId from a JWT token (convenience method).
     */
    fun extractUserId(token: String): String {
        return decode(token).userId
    }

    /**
     * Extracts userId from the JSON payload.
     * Tries user.publicUserId first, then user.userId as fallback,
     * then top-level "sub" or "userId" claims.
     */
    private fun extractUserId(jsonObject: JSONObject): String {
        // Try user.publicUserId first (per spec)
        if (jsonObject.has("user")) {
            val userObject = jsonObject.optJSONObject("user")
            if (userObject != null) {
                val publicUserId = userObject.optString("publicUserId", "")
                if (publicUserId.isNotEmpty()) {
                    return publicUserId
                }

                val userId = userObject.optString("userId", "")
                if (userId.isNotEmpty()) {
                    return userId
                }
            }
        }

        // Fallback to top-level claims (for compatibility)
        val sub = jsonObject.optString("sub", "")
        if (sub.isNotEmpty()) {
            return sub
        }

        val userId = jsonObject.optString("userId", "")
        if (userId.isNotEmpty()) {
            return userId
        }

        throw AmityException.create(
            "JWT token missing userId claim (tried 'user.publicUserId', 'user.userId', 'sub', 'userId')",
            null,
            AmityError.INVALID_PARAMETER
        )
    }

    private fun decodeBase64(encoded: String): String {
        // JWT uses Base64Url encoding, which may not have padding
        val base64 = encoded
            .replace("-", "+")
            .replace("_", "/")

        // Add padding if necessary
        val paddedBase64 = when (base64.length % 4) {
            2 -> "$base64=="
            3 -> "$base64="
            else -> base64
        }

        val decodedBytes = Base64.decode(paddedBase64, Base64.DEFAULT)
        return String(decodedBytes, Charsets.UTF_8)
    }
}
