package com.amity.socialcloud.sdk.core.session.component

import com.amity.socialcloud.sdk.core.CoreClient
import com.amity.socialcloud.sdk.core.domain.session.ActiveUserIdGetUseCase
import com.amity.socialcloud.sdk.core.session.AccessTokenRenewalImpl
import com.amity.socialcloud.sdk.core.session.eventbus.AppEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.core.session.model.AppEvent
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.amity.socialcloud.sdk.log.AmityLog
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.amity.socialcloud.sdk.model.core.session.LoginMethod
import com.amity.socialcloud.sdk.model.core.session.SessionHandler
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.joda.time.DateTime
import org.joda.time.Duration

class TokenRenewalSessionComponent(
    private val appEventBus: AppEventBus,
    sessionLifeCycleEventBus: SessionLifeCycleEventBus,
    sessionStateEventBus: SessionStateEventBus
) : SessionComponent(
    sessionLifeCycleEventBus, sessionStateEventBus
) {

    companion object {
        private const val TAG = "TokenRenewalSessionComponent"
    }

    var sessionHandler: SessionHandler? = null
    private var currentRenewal: AccessTokenRenewalImpl? = null

    //initiate the first one to be never failed
    private var lastFailedDateTime = DateTime.now().minus(Duration.standardDays(999))

    init {
        observeTokenEvent()
    }

    override fun onSessionStateChange(sessionState: SessionState) {
    }

    override fun establish(account: EkoAccount) {
    }

    override fun destroy() {
    }

    override fun handleTokenExpire() {
        finishCurrentRenewal()
    }

    private fun observeTokenEvent() {
        appEventBus.observe()
            .filter { it is AppEvent.TokenExpire || it is AppEvent.TokenAboutToExpire }
            .doOnNext { event -> initiateRenewalIfNeeded(isTokenExpired = event is AppEvent.TokenExpire) }
            .subscribe()
    }

    /**
     * Initiates token renewal based on login method and token state.
     * Per setAccessTokenHandler.md spec:
     * - REQ-005: Token expiry detection operates the same way regardless of login method
     * - REQ-006-013: Proactive renewal (about to expire)
     * - REQ-014-021: Expired token renewal
     *
     * @param isTokenExpired true if token has already expired, false if about to expire
     */
    private fun initiateRenewalIfNeeded(isTokenExpired: Boolean = false) {
        if (!canInitiateRenewal()) {
            return
        }

        // Check login method to determine which handler to use (REQ-006, REQ-014)
        val loginMethod = CoreClient.getLoginMethod()

        if (loginMethod == LoginMethod.ACCESS_TOKEN) {
            // Use AccessTokenHandler for access token login
            initiateAccessTokenHandlerRenewal(isTokenExpired)
        } else {
            // Use SessionHandler for userId login (existing flow)
            // Per REQ-007, REQ-015: If login method is NOT accessToken, skip handler invocation
            initiateSessionHandlerRenewal()
        }
    }

    /**
     * Initiates token renewal using AccessTokenHandler (for loginWithAccessToken sessions).
     * Per setAccessTokenHandler.md spec REQ-006 to REQ-021.
     *
     * @param isTokenExpired true if token has already expired (sets state to tokenExpired per REQ-018)
     */
    private fun initiateAccessTokenHandlerRenewal(isTokenExpired: Boolean) {
        val handler = CoreClient.accessTokenHandler

        // Per REQ-039: If loginMethod is accessToken and handler is not set, throw HANDLER_NOT_SET error
        if (handler == null) {
            AmityLog.e(TAG, "AccessTokenHandler not set for accessToken login. Per spec REQ-039: HANDLER_NOT_SET")
            // Per spec: throw error when handler not set for accessToken login
            val error = AmityException.create(
                "AccessTokenHandler not registered. Call setAccessTokenHandler() before loginWithAccessToken().",
                null,
                AmityError.HANDLER_NOT_SET
            )
            appEventBus.publish(AppEvent.TerminationCodeReceive(error))
            return
        }

        val userId = ActiveUserIdGetUseCase().execute()
        if (userId.isEmpty()) {
            AmityLog.e(TAG, "No active userId found. Token renewal skipped.")
            return
        }

        // Mark that we're attempting renewal
        currentRenewal = AccessTokenRenewalImpl(
            onRenewTokenSuccess = {
                appEventBus.publish(AppEvent.TokenRenewSuccess)
                finishCurrentRenewal()
            },
            onRenewTokenFailed = { throwable ->
                val amityError = AmityException.fromThrowable(throwable)
                if (amityError.code == AmityError.USER_IS_GLOBAL_BANNED.code ||
                    amityError.code == AmityError.UNAUTHORIZED_ERROR.code
                ) {
                    appEventBus.publish(AppEvent.TerminationCodeReceive(amityError))
                    finishCurrentRenewal()
                } else {
                    unableToRenewWithCurrentRenewal()
                }
            },
            onUnableToRetrieveAuthToken = {
                unableToRenewWithCurrentRenewal()
            },
            appEventBus = appEventBus,
            sessionLifeCycleEventBus = getSessionLifeCycleEventBus()
        )

        // Invoke handler asynchronously
        CoroutineScope(Dispatchers.IO).launch {
            try {
                val newAccessToken = handler.onTokenRenew(userId)

                // Use the new token to re-login
                CoreClient.loginWithAccessToken(newAccessToken)
                    .build()
                    .submit()
                    .doOnComplete {
                        appEventBus.publish(AppEvent.TokenRenewSuccess)
                        finishCurrentRenewal()
                    }
                    .doOnError { throwable ->
                        val amityError = AmityException.fromThrowable(throwable)
                        if (amityError.code == AmityError.USER_IS_GLOBAL_BANNED.code ||
                            amityError.code == AmityError.UNAUTHORIZED_ERROR.code
                        ) {
                            appEventBus.publish(AppEvent.TerminationCodeReceive(amityError))
                            finishCurrentRenewal()
                        } else {
                            unableToRenewWithCurrentRenewal()
                        }
                    }
                    .subscribe()
            } catch (e: Exception) {
                AmityLog.e(TAG, "AccessTokenHandler.onTokenRenew failed: ${e.message}")
                unableToRenewWithCurrentRenewal()
            }
        }
    }

    /**
     * Initiates token renewal using SessionHandler (for userId login sessions - existing flow)
     */
    private fun initiateSessionHandlerRenewal() {
        val accessTokenRenewal = AccessTokenRenewalImpl(
            onRenewTokenSuccess = {
                appEventBus.publish(AppEvent.TokenRenewSuccess)
                finishCurrentRenewal()
            },
            onRenewTokenFailed = { throwable ->
                val amityError = AmityException.fromThrowable(throwable)
                if (amityError.code == AmityError.USER_IS_GLOBAL_BANNED.code ||
                    amityError.code == AmityError.UNAUTHORIZED_ERROR.code
                ) {
                    appEventBus.publish(AppEvent.TerminationCodeReceive(amityError))
                    finishCurrentRenewal()
                } else {
                    unableToRenewWithCurrentRenewal()
                }
            },
            onUnableToRetrieveAuthToken = {
                unableToRenewWithCurrentRenewal()
            },
            appEventBus = appEventBus,
            sessionLifeCycleEventBus = getSessionLifeCycleEventBus()
        )
        initiateRenewal(accessTokenRenewal)
        sessionHandler?.sessionWillRenewAccessToken(accessTokenRenewal)
    }

    private fun initiateRenewal(renewal: AccessTokenRenewalImpl) {
        this.currentRenewal = renewal
    }

    private fun canInitiateRenewal(): Boolean {
        if (this.currentRenewal != null) {
            // Already initiated access token renewal flow.
            // The SDK is waiting for user to give the renewal response.
            return false
        }
        val tenMinutes = Duration.standardMinutes(10)
        if (this.currentRenewal != null
            && DateTime.now().isAfter(this.lastFailedDateTime.plus(tenMinutes))
        ) {
            // The new renewal can be triggered only after 10 minutes from last fail.
            return false
        }
        return true
    }

    private fun unableToRenewWithCurrentRenewal() {
        this.currentRenewal?.invalidate()
        this.currentRenewal = null
        this.lastFailedDateTime = DateTime.now()
    }

    private fun finishCurrentRenewal() {
        this.currentRenewal?.invalidate()
        this.currentRenewal = null
        this.lastFailedDateTime = DateTime.now().minus(Duration.standardDays(999))
    }
}