package com.amity.socialcloud.sdk.core.domain.session

import com.amity.socialcloud.sdk.core.CoreClient
import com.amity.socialcloud.sdk.core.LogoutProcessState
import com.amity.socialcloud.sdk.core.data.session.SessionRepository
import com.amity.socialcloud.sdk.core.session.eventbus.AppEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.model.AppEvent
import com.amity.socialcloud.sdk.core.session.model.SessionLifeCycle
import com.amity.socialcloud.sdk.core.util.JwtTokenDecoder
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers
import org.joda.time.DateTime
import java.util.concurrent.TimeUnit

/**
 * Use case for authenticating with a customer-provided JWT access token.
 *
 * This login flow (per spec):
 * 1. Decodes the JWT to extract userId, issuedAt, expiresAt (REQ-001 to REQ-005)
 * 2. Checks if same user is re-authenticating (no logout/state change needed) or different user (logout first)
 * 3. Verifies the token with the backend (REQ-007)
 * 4. Establishes the session with loginMethod = ACCESS_TOKEN (REQ-011)
 */
internal class LoginWithAccessTokenUseCase {

    fun execute(
        appEventBus: AppEventBus?,
        sessionLifeCycleEventBus: SessionLifeCycleEventBus?,
        accessToken: String,
        authToken: String?,
        displayName: String?
    ): Single<EkoAccount> {
        // Step 1: Decode JWT to get userId, issuedAt, expiresAt (REQ-001 to REQ-005)
        val decodedToken: JwtTokenDecoder.DecodedToken
        try {
            decodedToken = JwtTokenDecoder.decode(accessToken)
        } catch (e: Exception) {
            return Single.error(e)
        }

        val userId = decodedToken.userId
        val currentUserId = ActiveUserIdGetUseCase().execute()
        val isSameUser = currentUserId.isNotEmpty() && userId == currentUserId

        return if (isSameUser) {
            // Same user re-authentication (REQ-027 to REQ-035)
            // Session state remains 'established' - no logout cycle needed
            handleSameUserReAuth(
                appEventBus = appEventBus,
                sessionLifeCycleEventBus = sessionLifeCycleEventBus,
                accessToken = accessToken,
                authToken = authToken,
                displayName = displayName,
                userId = userId,
                issuedAt = decodedToken.issuedAt,
                expiresAt = decodedToken.expiresAt
            )
        } else {
            // Fresh login or different user re-authentication (REQ-015 to REQ-026)
            handleFreshOrDifferentUserLogin(
                appEventBus = appEventBus,
                sessionLifeCycleEventBus = sessionLifeCycleEventBus,
                accessToken = accessToken,
                authToken = authToken,
                displayName = displayName,
                userId = userId,
                issuedAt = decodedToken.issuedAt,
                expiresAt = decodedToken.expiresAt,
                currentUserId = currentUserId
            )
        }
    }

    /**
     * Handles same user re-authentication.
     * Per REQ-027-035: Session state remains 'established' throughout the process.
     */
    private fun handleSameUserReAuth(
        appEventBus: AppEventBus?,
        sessionLifeCycleEventBus: SessionLifeCycleEventBus?,
        accessToken: String,
        authToken: String?,
        displayName: String?,
        userId: String,
        issuedAt: DateTime?,
        expiresAt: DateTime?
    ): Single<EkoAccount> {
        // Verify token and update stored token without changing session state (REQ-031-033)
        return SessionRepository().loginWithAccessToken(
            userId = userId,
            displayName = displayName,
            accessToken = accessToken,
            authToken = authToken,
            issuedAt = issuedAt,
            expiresAt = expiresAt
        ).andThen(
            Single.defer { SessionRepository().getCurrentAccount() }
        ).doOnSuccess {
            // Don't change session state for same user (REQ-033)
            // Just ensure the session is established
            sessionLifeCycleEventBus?.publish(SessionLifeCycle.Establish(it))
        }.doOnError {
            val amityError = AmityException.fromThrowable(it)
            if (amityError.code == AmityError.USER_IS_GLOBAL_BANNED.code ||
                amityError.code == AmityError.UNAUTHORIZED_ERROR.code
            ) {
                appEventBus?.publish(AppEvent.TerminationCodeReceive(amityError))
            }
        }
    }

    /**
     * Handles fresh login or different user re-authentication.
     * Per REQ-015-026: Different user triggers logout, then establishing → established flow.
     */
    private fun handleFreshOrDifferentUserLogin(
        appEventBus: AppEventBus?,
        sessionLifeCycleEventBus: SessionLifeCycleEventBus?,
        accessToken: String,
        authToken: String?,
        displayName: String?,
        userId: String,
        issuedAt: DateTime?,
        expiresAt: DateTime?,
        currentUserId: String
    ): Single<EkoAccount> {
        val loginCompletable = verifyAndHandleUserChange(
            appEventBus = appEventBus,
            currentUserId = currentUserId
        )
            .andThen(Completable.defer { waitForLogoutCompletion() })
            .andThen(
                Completable.defer {
                    // Set state to establishing (REQ-006, REQ-021)
                    appEventBus?.publish(AppEvent.LoggingIn)
                    SessionRepository().loginWithAccessToken(
                        userId = userId,
                        displayName = displayName,
                        accessToken = accessToken,
                        authToken = authToken,
                        issuedAt = issuedAt,
                        expiresAt = expiresAt
                    )
                }
            )

        return handleLoggingIn(
            appEventBus = appEventBus,
            sessionLifeCycleEventBus = sessionLifeCycleEventBus,
            loginCompletable = loginCompletable
        )
    }

    private fun handleLoggingIn(
        appEventBus: AppEventBus?,
        sessionLifeCycleEventBus: SessionLifeCycleEventBus?,
        loginCompletable: Completable,
    ): Single<EkoAccount> {
        return loginCompletable.andThen(
            Single.defer { SessionRepository().getCurrentAccount() }
        ).doOnSuccess {
            // Set state to established (REQ-013, REQ-025)
            appEventBus?.publish(AppEvent.LoginSuccess)
            sessionLifeCycleEventBus?.publish(SessionLifeCycle.Establish(it))
        }.doOnError {
            val amityError = AmityException.fromThrowable(it)
            if (amityError.code == AmityError.USER_IS_GLOBAL_BANNED.code ||
                amityError.code == AmityError.UNAUTHORIZED_ERROR.code
            ) {
                appEventBus?.publish(AppEvent.TerminationCodeReceive(amityError))
            } else {
                // Set state to notLoggedIn on failure (REQ-009)
                appEventBus?.publish(AppEvent.LoginFail)
            }
        }
    }

    private fun waitForLogoutCompletion(): Completable {
        return Flowable.interval(100, TimeUnit.MILLISECONDS)
            .filter { CoreClient.logoutProcessState != LogoutProcessState.LOGGING_OUT }
            .firstElement()
            .ignoreElement()
            .subscribeOn(Schedulers.io())
    }

    /**
     * Handles different user scenario - triggers logout first (REQ-020).
     */
    private fun verifyAndHandleUserChange(
        appEventBus: AppEventBus?,
        currentUserId: String,
    ): Completable {
        return Completable.fromCallable {
            if (currentUserId.isNotEmpty()) {
                // Different user - need to logout current user first (REQ-020)
                appEventBus?.publish(AppEvent.ManualLogout)
            }
        }
    }
}
