package com.amity.socialcloud.sdk.core.data.session

import co.amity.rxbridge.toRx2
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.core.CoreClient
import com.amity.socialcloud.sdk.core.data.notification.device.DeviceNotificationRepository
import com.amity.socialcloud.sdk.core.data.user.UserQueryPersister
import com.amity.socialcloud.sdk.core.data.user.UserRepository
import com.amity.socialcloud.sdk.core.exception.EntityNotFoundException
import com.amity.socialcloud.sdk.log.AmityLog
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.amity.socialcloud.sdk.model.core.session.LoginMethod
import com.amity.socialcloud.sdk.model.core.user.AmityUserType
import com.amity.socialcloud.sdk.model.core.user.UserUpdateOption
import com.amity.socialcloud.sdk.model.core.user.isVisitor
import com.ekoapp.ekosdk.EkoObjectRepository
import com.ekoapp.ekosdk.internal.api.dto.EkoUserListDto
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import com.github.davidmoten.rx2.RetryWhen
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers
import org.joda.time.DateTime
import org.joda.time.Duration
import java.util.concurrent.TimeUnit

internal class SessionRepository : EkoObjectRepository() {

    private val ABOUT_TO_EXPIRE_TRESHOLD = 0.8

    fun getActiveUserId(): String {
        return SessionLocalDataStore().getActiveUserId()
    }

    fun getVisitorDeviceId(): String {
        return SessionLocalDataStore().getVisitorDeviceId()
    }

    fun getActiveUserType(): AmityUserType {
        return SessionLocalDataStore().getActiveUserType()
    }

    fun login(
        userId: String,
        displayName: String?,
        authToken: String?,
        isLegacyVersion: Boolean
    ): Completable {
        return activateAccount(userId)
            .flatMapCompletable { account ->
                // Set login method to USER_ID for traditional login
                account.loginMethod = LoginMethod.USER_ID.value
                renewTokenIfNeed(
                    account = account,
                    displayName = displayName,
                    authToken = authToken,
                    authSignature = null, // not needed for regular login
                    authSignatureExpiresAt = null, // not needed for regular login
                    isLegacyVersion = isLegacyVersion,
                )
            }
            .doOnComplete {
                // Do this here; to not break the stream when there is an error
                updateDisplayNameIfNeeded(userId, displayName)
            }
    }

    fun loginVisitor(
        displayName: String?,
        authSignature: String?,
        authSignatureExpiresAt: DateTime?,
    ): Completable {
        return activateVisitor()
            .flatMapCompletable {
                renewTokenIfNeed(
                    account = it,
                    displayName = displayName,
                    authToken = null, // not needed for visitor
                    authSignature = authSignature,
                    authSignatureExpiresAt = authSignatureExpiresAt,
                    isLegacyVersion = false,
                )
            }
    }

    /**
     * Login with a customer-provided JWT access token.
     *
     * This method (per spec REQ-001 to REQ-014):
     * 1. Activates the account for the given userId
     * 2. Verifies the access token with the backend
     * 3. Sets loginMethod to ACCESS_TOKEN
     * 4. Stores token with issuedAt and expiresAt
     * 5. Updates display name if provided
     */
    fun loginWithAccessToken(
        userId: String,
        displayName: String?,
        accessToken: String,
        authToken: String?,
        issuedAt: DateTime? = null,
        expiresAt: DateTime? = null
    ): Completable {
        return activateAccount(userId, accessToken)
            .flatMapCompletable { account ->
                // accessToken is already saved in DB via activateAccount
                account.loginMethod = LoginMethod.ACCESS_TOKEN.value

                // Verify token and register device
                renewTokenForAccessTokenLogin(
                    account = account,
                    displayName = displayName,
                    accessToken = accessToken,
                    authToken = authToken,
                    tokenIssuedAt = issuedAt,
                    tokenExpiresAt = expiresAt
                )
            }
            .doOnComplete {
                updateDisplayNameIfNeeded(userId, displayName)
            }
    }

    /**
     * Gets the current login method (userId or accessToken).
     * Returns null if no user is logged in.
     */
    fun getLoginMethod(): LoginMethod? {
        val account = SessionLocalDataStore().getCurrentAccount()
        return account?.loginMethod?.let { LoginMethod.fromValue(it) }
    }

    fun logout() : Completable{
        return Completable.fromAction {
            CoreClient.publishLogoutEvent()
        }
    }

    fun revokeAccessToken(): Single<Boolean> {
        return SessionRemoteDataStore().revokeAccessToken()
    }

    fun clearData(): Completable {
        return DeviceNotificationRepository().unregisterAll().andThen(
            Completable.fromAction {
                SessionLocalDataStore().logoutAccount(SessionLocalDataStore().getActiveUserId())
            }.subscribeOn(Schedulers.io())
        )
    }

    fun getCurrentAccount(): Single<EkoAccount> {
        return Single.create {
            val account = SessionLocalDataStore().getCurrentAccount()
            if (account != null) {
                it.onSuccess(account)
            } else {
                it.onError(EntityNotFoundException)
            }
        }.subscribeOn(Schedulers.io())
    }


    fun renewToken(
        account: EkoAccount,
        displayName: String? = null,
        authToken: String?,
        authSignature: String?,
        authSignatureExpiresAt: DateTime?,
        isLegacyVersion: Boolean,
    ): Completable {
        return SessionLocalDataStore().getApiKey()
            .flatMapCompletable {
                if (account.isVisitor()) {
                    SessionRemoteDataStore().registerVisitor(
                        apiKey = it.apiKey,
                        displayName = displayName,
                        deviceId = account.deviceId,
                        authSignature = authSignature,
                        authSignatureExpiresAt = authSignatureExpiresAt
                    )
                } else {
                    SessionRemoteDataStore().registerDevice(
                            apiKey = it.apiKey,
                            userId = account.userId,
                            displayName = displayName,
                            deviceId = account.deviceId,
                            authToken = authToken,
                            isLegacyVersion = isLegacyVersion
                    )
                }
                    .toRx2()
                    .retryWhen(
                        RetryWhen
                            .retryIf { AmityError.from(it) != AmityError.USER_IS_GLOBAL_BANNED }
                            .maxRetries(3)
                            .exponentialBackoff(1, 10, TimeUnit.SECONDS, 1.5)
                            .build()
                    ).toRx3()
                    .flatMapCompletable {
                        account.refreshToken = it.refreshToken
                        account.accessToken = it.accessToken
                        if (isLegacyVersion) {
                            //for legacy version, there's no expiration yet
                            account.issuedAt = DateTime.now()
                            account.expiresAt = DateTime.now().plus(Duration.standardDays(365))
                        } else {
                            account.issuedAt = it.issuedAt
                            account.expiresAt = it.expiresAt
                        }
                        val tokenDuration = (account.expiresAt.millis
                                - account.issuedAt.millis
                                - CoreClient.millisTimeDiff) * ABOUT_TO_EXPIRE_TRESHOLD
                        account.aboutToExpireAt = account.issuedAt.plus(tokenDuration.toLong())


                        AmityLog.e(
                                "SSM3", "token renewed: account updated : \n " +
                                "      expiresAt = ${account.expiresAt} \naboutToExpireAt = ${account.aboutToExpireAt}\n" +
                                "       issuedAt = ${account.issuedAt}" +
                                "\nduration: = ${tokenDuration / 1000} sec "
                        )

                        val users = EkoUserListDto().apply {
                            users = it.users ?: emptyList()
                            files = it.files ?: emptyList()
                        }
                        SessionLocalDataStore().updateAccount(account)
                                .andThen(UserQueryPersister().persist(users))
                    }
                    .doOnError {
                        SessionLocalDataStore().logoutAccount(account.userId)
                    }
            }
    }

    /**
     * Renews/verifies token for access token login.
     * Uses the provided access token directly instead of getting one from server.
     * Per spec REQ-004, REQ-005: Store issuedAt and expiresAt from JWT token.
     */
    private fun renewTokenForAccessTokenLogin(
        account: EkoAccount,
        displayName: String?,
        accessToken: String,
        authToken: String?,
        tokenIssuedAt: DateTime? = null,
        tokenExpiresAt: DateTime? = null
    ): Completable {
        return SessionLocalDataStore().getApiKey()
            .flatMapCompletable { apiKey ->
                // First verify the access token with the backend (REQ-007)
                SessionRemoteDataStore().verifyAccessToken(apiKey.apiKey, accessToken)
                    .andThen(
                        // Token verified, now register device with the access token
                        SessionRemoteDataStore().registerDevice(
                            apiKey = apiKey.apiKey,
                            userId = account.userId,
                            displayName = displayName,
                            deviceId = account.deviceId,
                            authToken = authToken,
                            isLegacyVersion = false
                        )
                            .toRx2()
                            .retryWhen(
                                RetryWhen
                                    .retryIf { AmityError.from(it) != AmityError.USER_IS_GLOBAL_BANNED }
                                    .maxRetries(3)
                                    .exponentialBackoff(1, 10, TimeUnit.SECONDS, 1.5)
                                    .build()
                            ).toRx3()
                            .flatMapCompletable { response ->
                                account.refreshToken = response.refreshToken
                                account.accessToken = response.accessToken

                                // Per spec REQ-004, REQ-005: Use timestamps from JWT if available,
                                // otherwise fallback to server response
                                account.issuedAt = tokenIssuedAt ?: response.issuedAt
                                account.expiresAt = tokenExpiresAt ?: response.expiresAt
                                account.loginMethod = LoginMethod.ACCESS_TOKEN.value

                                val tokenDuration = (account.expiresAt.millis
                                        - account.issuedAt.millis
                                        - CoreClient.millisTimeDiff) * ABOUT_TO_EXPIRE_TRESHOLD
                                account.aboutToExpireAt = account.issuedAt.plus(tokenDuration.toLong())

                                AmityLog.e(
                                    "SSM3", "access token login: account updated : \n " +
                                    "      expiresAt = ${account.expiresAt} \naboutToExpireAt = ${account.aboutToExpireAt}\n" +
                                    "       issuedAt = ${account.issuedAt}" +
                                    "\nduration: = ${tokenDuration / 1000} sec "
                                )

                                val users = EkoUserListDto().apply {
                                    users = response.users ?: emptyList()
                                    files = response.files ?: emptyList()
                                }
                                SessionLocalDataStore().updateAccount(account)
                                    .andThen(UserQueryPersister().persist(users))
                            }
                    )
                    .doOnError {
                        SessionLocalDataStore().logoutAccount(account.userId)
                    }
            }
    }

    private fun activateAccount(userId: String, accessToken: String? = null): Single<EkoAccount> {
        return Single.fromCallable {
            SessionLocalDataStore().activateAccount(
                userId = userId,
                accessToken = accessToken,
            )
        }.subscribeOn(Schedulers.io())
    }

    private fun activateVisitor(): Single<EkoAccount> {
        return Single.fromCallable {
            SessionLocalDataStore().activateVisitorAccount()
        }.subscribeOn(Schedulers.io())
    }

    private fun renewTokenIfNeed(
        account: EkoAccount,
        displayName: String?,
        authToken: String?,
        authSignature: String?,
        authSignatureExpiresAt: DateTime?,
        isLegacyVersion: Boolean
    ): Completable {
        return verifyCurrentAccessToken(account).onErrorResumeNext {
            if (AmityError.from(it) == AmityError.USER_IS_GLOBAL_BANNED) {
                Completable.error(it)
            } else {
                renewToken(account, displayName, authToken, authSignature, authSignatureExpiresAt, isLegacyVersion)
            }
        }
    }

    private fun verifyCurrentAccessToken(account: EkoAccount): Completable {
        return SessionLocalDataStore().getApiKey()
            .flatMapCompletable {
                if (account.accessToken == null) {
                    Completable.error(
                        AmityException.create(
                            "accessToken not found",
                            null,
                            AmityError.UNAUTHORIZED_ERROR
                        )
                    )
                } else {
                    SessionRemoteDataStore().verifyAccessToken(it.apiKey)
                }
            }
    }

    private fun updateDisplayNameIfNeeded(userId: String, displayName: String?) {
        if (displayName != null) {
            UserRepository().observe(userId)
                .firstOrError()
                .flatMapCompletable {
                    if (it.getDisplayName() == displayName) {
                        Completable.complete()
                    } else {
                        UserRepository().updateUser(
                            userId,
                            UserUpdateOption(displayName = displayName)
                        )
                            .ignoreElement()
                    }
                }
                .subscribeOn(Schedulers.io())
                .doOnError {
                    // do nothing
                }
                .subscribe()
        }
    }

}