package com.amity.socialcloud.sdk.api.social.post.update

import com.amity.socialcloud.sdk.model.core.file.AmityAudio
import com.amity.socialcloud.sdk.model.core.file.AmityFile
import com.amity.socialcloud.sdk.model.core.file.AmityFileInfo
import com.amity.socialcloud.sdk.model.core.file.AmityImage
import com.amity.socialcloud.sdk.model.core.file.AmityAttachment
import com.amity.socialcloud.sdk.model.core.file.AmityVideo
import com.amity.socialcloud.sdk.model.core.link.AmityLink
import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.amity.socialcloud.sdk.model.social.post.AmityPostAttachment
import com.amity.socialcloud.sdk.social.domain.post.update.PostEditUseCase
import com.ekoapp.ekosdk.internal.usecase.post.PostCreateUseCase
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable


class AmityPostUpdate internal constructor(
    private val postId: String,
    private val title: String?,
    private val text: String?,
    private val metadata: JsonObject?,
    private val mentionees: List<String>?,
    private val attachments: List<AmityPostAttachment>?,
    private val hashtags: List<String>?,
    private val tags: List<String>?,
    private val links: List<AmityLink>?,
)  {

    private fun getData(): JsonObject {
        return JsonObject().apply {
            text?.let { addProperty("text", it) }
            title?.let { addProperty("title", it) }
        }
    }

    /**
     * Updates a post
     * @return Async post as Single<AmityPost>
     */
    fun apply(): Completable {
        return PostEditUseCase().execute(postId, getData(), metadata, mentionees, attachments, hashtags, tags, links)
    }

    class Builder internal constructor(private val postId: String) {

        private var text: String? = null
        private var title: String? = null
        private var metadata: JsonObject? = null
        private var mentionees: List<String>? = null
        private var attachments: List<AmityPostAttachment>? = null
        private var hashtags: List<String>? = null
        private var tags: List<String>? = null
        private var links: List<AmityLink>? = null

        /**
         * @param [text] to replace text on the post
         * @return the same instance of AmityPostUpdate.Builder
         */
        fun text(text: String): Builder {
            return apply {
                this.text = text
            }
        }

        /**
         * @param [title] to replace title on the post
         * @return the same instance of AmityPostUpdate.Builder
         */
        fun title(title: String): Builder {
            return apply {
                this.title = title
            }
        }

        /**
         * @param [attachments] to replace video children
         * @return the same instance of AmityPostUpdate.Builder
         */
        fun attachments(vararg attachments: AmityVideo): Builder {
            this.attachments = attachments.toList().map { AmityPostAttachment(it.getFileId(),  AmityPost.DataType.VIDEO.getApiKey()) }
            return this
        }

        /**
         * @param [attachments] to replace image children
         * @return the same instance of AmityPostUpdate.Builder
         */
        fun attachments(vararg attachments: AmityImage): Builder {
            this.attachments = attachments.toList().map { AmityPostAttachment(it.getFileId(),  AmityPost.DataType.IMAGE.getApiKey()) }
            return this
        }

        /**
         * @param [attachments] to replace file children
         * @return the same instance of AmityPostUpdate.Builder
         */
        fun attachments(vararg attachments: AmityFile): Builder {
            this.attachments = attachments.toList().map { AmityPostAttachment(it.getFileId(),  AmityPost.DataType.FILE.getApiKey()) }
            return this
        }

        fun attachments(vararg attachments: AmityAudio): Builder {
            this.attachments = attachments.toList().map { AmityPostAttachment(it.getFileId(),  AmityPost.DataType.AUDIO.getApiKey()) }
            return this
        }

        fun attachments(vararg attachments: AmityAttachment): Builder {
            this.attachments = attachments.toList().map {
                (it as AmityFileInfo)
                AmityPostAttachment(it.getFileId(), it.rawFile.getFileType().apiKey)
            }
            return this
        }

        @JvmName("attachmentList")
        fun attachments(attachments: List<AmityAttachment>): Builder {
            this.attachments = attachments.map {
                (it as AmityFileInfo)
                AmityPostAttachment(it.getFileId(), it.rawFile.getFileType().apiKey)
            }
            return this
        }

        @JvmName("images")
        fun attachments(attachments: List<AmityImage>): Builder {
            this.attachments = attachments.map {
                (it as AmityFileInfo)
                AmityPostAttachment(it.getFileId(),  AmityPost.DataType.IMAGE.getApiKey())
            }
            return this
        }

        @JvmName("videos")
        fun attachments(attachments: List<AmityVideo>): Builder {
            this.attachments = attachments.map {
                (it as AmityFileInfo)
                AmityPostAttachment(it.getFileId(),  AmityPost.DataType.VIDEO.getApiKey())
            }
            return this
        }

        @JvmName("audios")
        fun attachments(attachments: List<AmityAudio>): Builder {
            this.attachments = attachments.map {
                (it as AmityFileInfo)
                AmityPostAttachment(it.getFileId(),  AmityPost.DataType.AUDIO.getApiKey())
            }
            return this
        }

        @JvmName("files")
        fun attachments(attachments: List<AmityFile>): Builder {
            this.attachments = attachments.map {
                (it as AmityFileInfo)
                AmityPostAttachment(it.getFileId(),  AmityPost.DataType.FILE.getApiKey())
            }
            return this
        }

        /**
         * @param [metadata] to replace metadata
         * @return the same instance of AmityPostUpdate.Builder
         */
        fun metadata(metadata: JsonObject): Builder {
            this.metadata = metadata
            return this
        }

        /**
         * @param [userIds] to update the list of userId of users to be mentioned by this post
         * @return the same instance of AmityPostUpdate.Builder
         */
        fun mentionUsers(userIds: List<String>): Builder {
            this.mentionees = userIds
            return this
        }

        /**
         * @param [hashtags] to update the list of hashtags for this post
         * @return the same instance of AmityPostUpdate.Builder
         */
        fun hashtags(hashtags: List<String>): Builder {
            this.hashtags = hashtags
            return this
        }

        /**
         * @param [tags] to update the list of tags for this post
         * @return the same instance of AmityPostUpdate.Builder
         */
        fun tags(tags: List<String>): Builder {
            this.tags = tags
            return this
        }

        /**
         * @param [links] to update the list of links for this post
         * @return the same instance of AmityPostUpdate.Builder
         */
        fun links(links: List<AmityLink>): Builder {
            this.links = links
            return this
        }

        /**
         * Instantiates AmityPostUpdate with built params
         * @return new instance of AmityPostUpdate
         */
        fun build(): AmityPostUpdate {
            return AmityPostUpdate(postId, title, text, metadata, mentionees, attachments, hashtags, tags, links)
        }

    }

}