package com.amity.socialcloud.sdk.api.social.post.query

import androidx.paging.PagingData
import com.amity.socialcloud.sdk.api.social.post.review.AmityReviewStatus
import com.amity.socialcloud.sdk.model.social.feed.AmityFeedType
import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.ekoapp.ekosdk.internal.usecase.post.GetCommunityPostPagingDataUseCase
import io.reactivex.rxjava3.core.Flowable

class AmityCommunityPostQuery internal constructor(
    private val communityId: String,
    private val sortOption: AmityCommunityFeedSortOption,
    private val isDeleted: Boolean?,
    private val feedType: AmityFeedType,
    @Deprecated("Use dataTypes instead")
    private val postTypes: List<AmityPost.DataType>,
    private val dataTypes: List<AmityPost.DataType> = postTypes,
    private val tags: List<String>,
    private val matchingOnlyParentPosts: Boolean?,
    private val includeMixedStructure: Boolean
) {

    /**
     * Queries for Community feed.
     * @return Flowable<PagingData<AmityPost>>.
     */
    fun query(): Flowable<PagingData<AmityPost>> {
        return GetCommunityPostPagingDataUseCase().execute(
            communityId = communityId,
            sortOption = sortOption,
            isDeleted = isDeleted,
            feedType = feedType,
            dataTypes = dataTypes.ifEmpty { postTypes },
            tags = tags,
            matchingOnlyParentPosts = matchingOnlyParentPosts ?: dataTypes.isEmpty() || dataTypes.contains(AmityPost.DataType.TEXT),
            includeMixedStructure = includeMixedStructure
        )
    }

    class Builder internal constructor(private val communityId: String) {

        private var sortOption: AmityCommunityFeedSortOption =
            AmityCommunityFeedSortOption.LAST_CREATED
        private var isDeleted: Boolean? = null
        private var feedType: AmityFeedType = AmityFeedType.PUBLISHED
        private var postTypes: List<AmityPost.DataType> = listOf()
        private var dataTypes: List<AmityPost.DataType> = listOf()
        private var tags: List<String> = listOf()
        private var matchingOnlyParentPosts: Boolean? = null
        private var includeMixedStructure: Boolean = false

        /**
         * Specifies [sortOption] param.
         * When [sortOption] is AmityCommunityFeedSortOption.FIRST_CREATED, posts will be sorted by createdAt property ASC.
         * When [sortOption] is AmityCommunityFeedSortOption.LAST_CREATED, posts will be sorted by createdAt property DESC.
         * The default value of [sortOption] is AmityCommunityFeedSortOption.LAST_CREATED.
         * @return the same AmityCommunityPostQuery.Builder instance.
         */
        fun sortBy(sortOption: AmityCommunityFeedSortOption): Builder {
            return apply { this.sortOption = sortOption }
        }

        /**
         * Specifies [includeDeleted] param.
         * When [includeDeleted] is true, the result will include soft-deleted posts.
         * When [includeDeleted] is false, the result will exclude soft-deleted posts.
         * The default value of [includeDeleted] is true.
         * @return the same AmityCommunityPostQuery.Builder instance.
         */
        fun includeDeleted(includeDeleted: Boolean): Builder {
            return apply {
                this.isDeleted = if(includeDeleted) null else false
            }
        }

        @Deprecated("Use reviewStatus() instead")
        fun feedType(feedType: AmityFeedType): Builder {
            return apply { this.feedType = feedType }
        }

        /**
         * Specifies [reviewStatus] to filter matching reviewStatus.
         * @return the same AmityCommunityPostQuery.Builder instance.
         */
        fun reviewStatus(reviewStatus: AmityReviewStatus): Builder {
            return apply { this.feedType = AmityFeedType.enumOf(reviewStatus.apiKey) }
        }

        /**
         * Specifies [postTypes] to filter matching dataType.
         * @return the same AmityCommunityPostQuery.Builder instance.
         */
        @Deprecated("Use types(dataTypes) instead")
        fun types(postTypes: List<AmityPost.DataType>): Builder {
            return apply {
                this.postTypes = postTypes
                this.dataTypes = postTypes
            }
        }


        /**
         * Specifies [dataTypes] to filter matching dataType.
         * @return the same AmityUserPostQuery.Builder instance.
         */
        fun dataTypes(dataTypes: List<AmityPost.DataType>): Builder {
            return apply { this.dataTypes = dataTypes }
        }

        /**
         * Specifies [tags] to filter posts by tags.
         * @param tags Array of tag strings to filter posts (max 5)
         * @return the same AmityCommunityPostQuery.Builder instance.
         */
        fun tags(tags: List<String>): Builder {
            return apply { this.tags = tags.take(5) }
        }

        fun matchingOnlyParentPosts(matchingOnlyParentPosts: Boolean): Builder {
            return apply {
                this.matchingOnlyParentPosts = matchingOnlyParentPosts
            }
        }

        fun includeMixedStructure(includeMixedStructure: Boolean): Builder {
            return apply { this.includeMixedStructure = includeMixedStructure }
        }

        /**
         * Instantiates AmityCommunityPostQuery with built params.
         * @return new instance of AmityCommunityPostQuery.
         */
        fun build(): AmityCommunityPostQuery {
            return AmityCommunityPostQuery(
                communityId = communityId,
                sortOption = sortOption,
                isDeleted = isDeleted,
                feedType = feedType,
                postTypes = postTypes,
                dataTypes = dataTypes,
                tags = tags,
                matchingOnlyParentPosts = matchingOnlyParentPosts,
                includeMixedStructure = includeMixedStructure
            )
        }
    }

}