package com.amity.socialcloud.sdk.api.core.session

import com.amity.socialcloud.sdk.core.domain.session.LoginWithAccessTokenUseCase
import com.amity.socialcloud.sdk.core.session.eventbus.AppEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import io.reactivex.rxjava3.core.Completable

/**
 * Session establisher for login with customer-provided JWT access token.
 *
 * This establisher allows authentication using a pre-existing JWT access token
 * managed by the customer's backend, without requiring the userId to be provided
 * separately (it's extracted from the token).
 *
 * Usage:
 * ```kotlin
 * AmityCoreClient.loginWithAccessToken(accessToken)
 *     .displayName("John Doe")  // optional
 *     .build()
 *     .submit()
 *     .subscribe()
 * ```
 */
class AmityAccessTokenEstablisher private constructor(
    private val appEventBus: AppEventBus?,
    private val sessionLifeCycleEventBus: SessionLifeCycleEventBus?,
    private val accessToken: String,
    private val authToken: String?,
    private val displayName: String?
) {

    /**
     * Submits the login request with the access token.
     *
     * @return Completable that completes when login is successful
     */
    fun submit(): Completable {
        return LoginWithAccessTokenUseCase().execute(
            appEventBus = appEventBus,
            sessionLifeCycleEventBus = sessionLifeCycleEventBus,
            accessToken = accessToken,
            authToken = authToken,
            displayName = displayName
        ).ignoreElement()
    }

    /**
     * Builder for creating an AmityAccessTokenEstablisher.
     */
    class Builder internal constructor(
        private val appEventBus: AppEventBus?,
        private val sessionLifeCycleEventBus: SessionLifeCycleEventBus?,
        private val accessToken: String
    ) {

        private var displayName: String? = null
        private var authToken: String? = null

        /**
         * Sets the display name for the user.
         * This will update the user's display name if it differs from the current one.
         *
         * @param displayName The display name to set
         * @return This builder for chaining
         */
        fun displayName(displayName: String): Builder {
            this.displayName = displayName
            return this
        }

        /**
         * Sets the auth token for the user on secure mode.
         *
         * @param authToken The auth token to set
         * @return This builder for chaining
         */
        fun authToken(authToken: String): Builder {
            this.authToken = authToken
            return this
        }

        /**
         * Builds the AmityAccessTokenEstablisher.
         *
         * @return The configured AmityAccessTokenEstablisher
         */
        fun build(): AmityAccessTokenEstablisher {
            return AmityAccessTokenEstablisher(
                appEventBus = appEventBus,
                sessionLifeCycleEventBus = sessionLifeCycleEventBus,
                accessToken = accessToken,
                authToken = authToken,
                displayName = displayName,
            )
        }
    }
}
