package com.ekoapp.ekosdk.internal.api.http.request

import android.util.Base64
import com.amity.socialcloud.sdk.entity.video.watch.WatchSessionEntity
import com.ekoapp.ekosdk.internal.api.EkoEndpoint
import org.joda.time.DateTime
import org.joda.time.DateTimeZone
import java.nio.charset.StandardCharsets
import java.security.KeyFactory
import java.security.PrivateKey
import java.security.Signature
import java.security.spec.PKCS8EncodedKeySpec
import java.util.*

internal class WatchSessionRequestCreator(private val watchSessionEntityList: List<WatchSessionEntity>) {

    fun create(): WatchSessionRequest {
        val watchSessionList = getWatchSessionList()
        val nonceString = generateNonceString()
        val timeStampString = DateTime.now().withZone(DateTimeZone.UTC).toString()
        val dataString = createDataString(watchSessionList)
        val unsignedString = composeUnsignedString(nonceString = nonceString,
                timeStampString = timeStampString, dataString = dataString)
        val signedString = createSignature(unsignedString)
        return WatchSessionRequest.create(signedString, nonceString, timeStampString, watchSessionList)
    }

    private fun getWatchSessionList(): List<WatchSession> {
        val watchSessionList = arrayListOf<WatchSession>()
        watchSessionEntityList.forEach { entity ->
            val startTime = entity.startTime?.withZone(DateTimeZone.UTC)?.toDateTimeISO()?.toString()
            val endTime = entity.endTime?.withZone(DateTimeZone.UTC)?.toString()
            if (startTime != null && endTime != null) {
                val watchSession = WatchSession.create(entity.sessionId, entity.roomId,
                        startTime, endTime, entity.watchSeconds)
                watchSessionList.add(watchSession)
            }
        }
        return watchSessionList.sortedBy { it.startTime }
    }

    private fun createDataString(watchSessionList: List<WatchSession>): String {
        val accumulatedString = StringBuilder()
        watchSessionList
                .forEachIndexed { index, watchSession ->
                    val isLastIndex = index == watchSessionList.size - 1
                    val sessionStringData = StringBuilder().append("endTime=${watchSession.endTime}&")
                            .append("roomId=${watchSession.roomId}&")
                            .append("sessionId=${watchSession.sessionId}&")
                            .append("startTime=${watchSession.startTime}&")
                            .append("watchSeconds=${watchSession.watchSeconds}")

                    accumulatedString.append(sessionStringData)

                    if (!isLastIndex) {
                        accumulatedString.append(";")
                    }
                }
        return accumulatedString.toString()
    }


    private fun composeUnsignedString(nonceString: String, timeStampString: String, dataString: String): String {
        return StringBuilder().append("nonceStr=$nonceString&")
                .append("timestamp=$timeStampString&")
                .append("data=$dataString==")
                .toString()
    }

    @Throws(Exception::class)
    private fun createSignature(plainText: String): String? {
        val privateSignature = Signature.getInstance("SHA256withRSA")
        privateSignature.initSign(getPrivateKey())
        privateSignature.update(plainText.toByteArray(StandardCharsets.UTF_8))
        val signature = privateSignature.sign()
        return Base64.encodeToString(signature, Base64.DEFAULT).replace("\n", "")
    }

    private fun getPrivateKey(): PrivateKey {
        val keyBytes: ByteArray = Base64.decode(getSecretKey(), Base64.DEFAULT)
        val keySpec = PKCS8EncodedKeySpec(keyBytes)
        val fact = KeyFactory.getInstance("RSA")
        return fact.generatePrivate(keySpec)
    }

    private fun generateNonceString(): String {
        return UUID.randomUUID().toString()
    }

    private fun getSecretKey(): String {
        return if (EkoEndpoint.getHttpUrl() == "https://api.staging.amity.co/"
                || EkoEndpoint.getHttpUrl() == "https://api.dev.amity.co/") {
            STAGING_SECRET_KEY
        } else {
            PROD_SECRET_KEY
        }
    }
}

private const val STAGING_SECRET_KEY = "MIIEpQIBAAKCAQEAwAEc/oZgYIvKSUG/C3mONYLR4ZPgAjMEX4bJ+xqqakUDRtql\n" +
        "NO+eZs2blQ1Ko0DBkqPExyQezvjibH5W2UZBV5RaBTlTcNVKTToMBEGesAfaEcM3\n" +
        "qUyQHxdbFYZv6P4sb14dcwxTQ8usmaV8ooiR1Fcaso5ZWYcZ8Hb46FbQ7OoVumsB\n" +
        "tPWwfZ4f003o5VCl6AIM6lcLv9UDLlFVYhE+PeXpRHtfWlGqxMvqC9oinlwhL6nW\n" +
        "v6VjQXW4nhcib72dPBzfHT7k/PMKto2SxALYdb68ENiAGuJLWi3AUHSyYCJK2w7w\n" +
        "IlWfJUAI0v26ub10IpExr6D5QuW2577jjP93iwIDAQABAoIBAFWfqXhwIIatkFY+\n" +
        "9Z1+ZcbDQimgsmMIsUiQaX6Lk7e0cxOj6czDlxYtVtaPiNtow2pLkjNkjkCqiP7t\n" +
        "EHnwdK9DvylZOTa2R15NJpK3WLcTqVIGhsn/FL5owfvFah6zSsmXZParZm5zY9NZ\n" +
        "E03ALZhOB9/cz0e3kf/EbpfeL2mW7MApyiUt5i09ycchroOpcWp73ipIxvgigtZy\n" +
        "UGFmsQicWhUs28F0D7w4Qfk76yG3nqXeb+BAMhCaIaa/k/aAxhiZG/ygEQWQrcC8\n" +
        "gfe+jyicMAQPDEVS9YuUMGsLjIjKuVLZzp2xirQnhc2i2zVNEIvG6soprPOBEMQu\n" +
        "gzrtX5ECgYEA3b7KAbBIbDl1e4ZSCWhHdHkiWVZHaopsR/LhqDDNhXjWjq3AesgV\n" +
        "6k0j9EdziMn/HmmOso0bz99GTV3JZf4A9ztTLumJlkHbdVtlgOqSjrFLj12rH9KX\n" +
        "TheyIhWSpUmm8+WB1xasFbqpvJaGo7F3pd2Fqj1XR4mp5BO7c/t7LJ0CgYEA3aou\n" +
        "EzXQ9THRKYocdfY69EI1Il1t/d/RSqqd9BxEjxBgxkM13ZiYIn/R4WW/nCUrlmhx\n" +
        "G44Aa2Gob4Ahfsui2xKTg/g/3Zk/rAxAEGkfOLGoenaJMD41fH4wUq3FRYwkvnaM\n" +
        "b9Hd6f/TlBHslIRa2NN58bSBGJCyBP2b59+2+EcCgYEAixDVRXvV37GlYUOa/XVd\n" +
        "osk5Zoe6oDGRuQm0xbNdoUBoZvDHDvme7ONWEiQha/8qtVsD+CyQ7awcPfb8kK9c\n" +
        "0bBt+bTS6d4BkTcxkEkMgtrkBVR8Nqfu5jXsLH4VCv4G61zbMhZw8+ut+az5YX2y\n" +
        "CN7Frj9sFlxapMRPQmzMEe0CgYEAumsAzM8ZqNv4mAK65Mnr0rhLj1cbxcKRdUYA\n" +
        "COgtEFQpzxN/HZnTeFAe5nx3pI3uFlRHq3DFEYnT6dHMWaJQmAULYpVIwMi9L6gt\n" +
        "yJ9fzoI6uqMtxRDMUqKdaSsTGOY/kJ6KhQ/unXi1K3XXjR+yd1+C0q+HUm1+CYxv\n" +
        "rZYLfskCgYEArsEy+IQOiqniJ0NE2vVUF+UK/IRZaic9YKcpov5Ot7Vvzm/MnnW4\n" +
        "N1ljVskocETBWMmPUvNSExVjPebi+rxd8fa5kY8BJScPTzMFbunZn/wjtGdcM10q\n" +
        "dlVQ9doG61A/9P3ezFKCfS4AvF/H/59LcSx2Bh28fp3/efiVIOpVd4Y="

private const val PROD_SECRET_KEY = "MIIEowIBAAKCAQEAx6PNEnnB+xbhdoRWJ2/pduv1TfFDC1wEBS58XIjUxMJBlRTK\n" +
        "XXs5R/K1Nh7VCS+ERE6IGwDi1WNFzI7OrD5/BtZwC5ofang6VLVahTlbQR8RGfdd\n" +
        "EyzrUIrp46iSjuZ1name9RAtzhZbkYhkJDvOAYiBW79+7m+8Zd0vF2q27SqC3Mla\n" +
        "ZAx4vtcMdaYAgJ+oLQVIHJDy+TUfIy7Hmm5dxwF0H4UwiqOLZjhTNg8cJdSFDJiC\n" +
        "ayYP2iFP1NLY/9RKKO5htCXDVl9fZVtxHnAm4tJ3AiTq5FHcJQa3I2jGsml/7mxM\n" +
        "MlHkxoi3cnOC9xkSx48ivMYFTV0TB22VidmuTQIDAQABAoIBAEsbhPt+c0xTv18n\n" +
        "SoJryBnFi/RW4DJV2+1XH9JwaBveOW4s9DVXNqWq2iISV3OtPMW2Bd6i5rQBNTFQ\n" +
        "SNDkckXQRzcWNNuJf+h+kh+tBHfriDOMrKeaLq2OlY/gy8KXUYiPWA46mAEAyFJo\n" +
        "TcST+1z4TjKGSOrLfsWUATN+InP27wDEiZvTmaYUqd/ql1ZBWc7a9psQ9hGUiSS2\n" +
        "MYvAQnYDNqoWs2pW3CeswyhwmVlVpvEUcQxpXcjG/9t+ycRE8z+NLQJczmQ5gQG1\n" +
        "IBJ7qwyfd0VQyBN622VLIcIHkHdfGnptPeploLwdQCCmILEwEd6cPldfWFgI6v4J\n" +
        "T+n8/MECgYEA/6EKvrpChdD0YAXjOYHk40sLHRqaUC5n+Z3bo76Is/06XYwxD7uk\n" +
        "fJIoEK9axtreCv8n3WaCPEXZiQ9Wpd0OzfYaH6KEsuK4+hN7cc1mQhyYx2ndnJwq\n" +
        "fztuh4kXhgJe8E3H05jXYhR4WUM5uSQZL1gIsLr0DDXxG518cOv7yA8CgYEAx+31\n" +
        "/ILHO+yS7wFoKdTGkDf7G6lW+iXk9RUH0fBP1hQkps0AbTc3eSoGLgnnpcK//A8m\n" +
        "MrbzdgcwmGV6e94CNYxd2JVuY1Z++DmOk8ENPeXYoMKOmdMrsaKXSDgKSoAJC/T0\n" +
        "SoDa6gH9zQMjLmupThon84w6IwtylH8pUJsTJ+MCgYB6dgfTvx9HRAcOSUPFtPiV\n" +
        "+slkM+5hIxokdv9Vvl12/IUNc6+9PcNa+hm/hIIUaZ2d3waWBhm2DjBcfMChzPje\n" +
        "qltxdEXl/L9o7hRhPGkUmuaGAa1s7VNeiHY27/IY/OHGdTyZncrw7aWJ/a6VhzmZ\n" +
        "eK8RGz726pCLDjMjVanU+QKBgDkoHpsCCFNpXthVXqIW4m14k6Fk2CjLjJlEWef6\n" +
        "+9gICjJliSmwD3vkN8ARtNf+xUQaUKFPDGtBwvLOkCW5YohMiMfScKyVndb5HXnA\n" +
        "Rzja7QOIm5v+cw9lnDpZSxfhYrS4Yt6p1+vuhdCGhRhQTya085b9D+HU8/zgHb+d\n" +
        "lexhAoGBANk7bTigQKlpZYDmmTpKpS1TCVZoEe7OrS4tfnBD/GXGqVfWkW0c0wCl\n" +
        "5uH/ABIbZRvRfZAjrx2AmBouiWoliWZB43nsd0/nhOg4usuQZvyH7eY7VYSeIwlc\n" +
        "BzikzHo+/TT1BUyX7/IQqNvYWgQ7+kVxa7/k9tSgnMPJN7CXiVlk"
