package com.amity.socialcloud.sdk.api.video.room.analytics

import com.amity.socialcloud.sdk.model.video.room.AmityRoom
import com.amity.socialcloud.sdk.video.domain.watch.CreateWatchSessionUseCase
import com.amity.socialcloud.sdk.video.domain.watch.SyncPendingWatchSessionsUseCase
import com.amity.socialcloud.sdk.video.domain.watch.UpdateWatchSessionUseCase
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Single
import org.joda.time.DateTime

/**
 * Analytics class for managing room watch sessions
 */
class AmityRoomAnalytics internal constructor(private val room: AmityRoom) {

    /**
     * Create a new watch session for the current room
     * @param startedAt The timestamp when watching started
     * @return Single<String> sessionId unique identifier for this watch session
     * @throws AmityException if room is not in watchable state (not LIVE or RECORDED)
     */
    fun createWatchSession(startedAt: DateTime): Single<String> {
        return CreateWatchSessionUseCase().execute(room, startedAt)
    }

    /**
     * Update an existing watch session with duration
     * @param sessionId The unique identifier of the watch session
     * @param duration The total watch duration in seconds
     * @param endedAt The timestamp when this update occurred
     * @return Completable Completion status
     */
    fun updateWatchSession(
        sessionId: String,
        duration: Long,
        endedAt: DateTime
    ): Completable {
        return UpdateWatchSessionUseCase().execute(sessionId, duration, endedAt)
    }

    /**
     * Sync all pending watch sessions to backend
     * This function uses jitter delay and handles network resilience
     */
    fun syncPendingWatchSessions() {
        SyncPendingWatchSessionsUseCase().execute()
    }
}
