package com.amity.socialcloud.sdk.video.domain.watch

import com.amity.socialcloud.sdk.entity.video.watch.WatchSessionEntity
import com.amity.socialcloud.sdk.video.data.watch.WatchSessionRepository
import com.ekoapp.ekosdk.AmityNetworkUtils
import com.ekoapp.ekosdk.internal.api.AmityHttpClient
import com.ekoapp.ekosdk.internal.api.EkoStreamApi
import com.ekoapp.ekosdk.internal.api.http.request.WatchSessionRequestCreator
import com.ekoapp.ekosdk.internal.data.EkoDatabase
import com.ekoapp.ekosdk.internal.data.model.EkoApiKey
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Maybe
import java.util.concurrent.TimeUnit

private const val ONE_MINUTE = 60

class SendWatchSessionUseCase {

    private var retryCount = 0

    fun execute(watchSessions: List<WatchSessionEntity>): Completable {
        return getApiKey()
                .flatMapCompletable { sendToServer(it.apiKey, watchSessions) }
                .doOnComplete { markAsSynced(watchSessions) }
                .doOnError { markAsFailed(watchSessions) }
    }

    fun execute(watchSessions: List<WatchSessionEntity>, requireNetwork: Boolean): Completable {
        return when {
            watchSessions.isEmpty() -> {
                Completable.complete()
            }
            requireNetwork -> {
                Flowable.interval(1, TimeUnit.SECONDS)
                        .filter { shouldSend() }
                        .takeUntil { retryCount >= ONE_MINUTE }
                        .firstOrError()
                        .flatMapCompletable { execute(watchSessions) }
            }
            else -> {
                execute(watchSessions)
            }
        }
    }

    private fun shouldSend(): Boolean {
        return if (AmityNetworkUtils.isNetworkAvailable()) {
            true
        } else {
            retryCount++
            false
        }
    }

    private fun getApiKey(): Maybe<EkoApiKey> {
        return EkoDatabase.get().apiKeyDao().currentApiKey
    }

    private fun sendToServer(apiKey: String, watchSessions: List<WatchSessionEntity>): Completable {
        WatchSessionRepository().markAsSyncing(watchSessions)
        val request = WatchSessionRequestCreator(watchSessions).create()
        return AmityHttpClient.get(EkoStreamApi::class).flatMapCompletable {
            it.sendWatchSession(apiKey, request)
        }
    }

    private fun markAsSynced(watchSessions: List<WatchSessionEntity>) {
        WatchSessionRepository().markAsSynced(watchSessions)
    }

    private fun markAsFailed(watchSessions: List<WatchSessionEntity>) {
        WatchSessionRepository().markAsFailed(watchSessions)
    }
}
