package com.amity.socialcloud.sdk.api.video.room

import com.amity.socialcloud.sdk.api.video.room.query.AmityRoomQuery
import com.amity.socialcloud.sdk.core.session.eventbus.CoHostEventBus
import com.amity.socialcloud.sdk.core.session.model.AmityCoHostEvent
import com.amity.socialcloud.sdk.model.video.room.AmityRoom
import com.amity.socialcloud.sdk.model.video.room.AmityRoomBroadcastData
import com.amity.socialcloud.sdk.video.domain.room.*
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers

class AmityRoomRepository internal constructor() {

    /* begin_public_function
      id: room.query
      api_style: live_collection
    */
    fun getRooms(): AmityRoomQuery.Builder = AmityRoomQuery.Builder()
    /* end_public_function */

    /* begin_public_function
      id: room.get
      api_style: live_object
    */
    fun getRoom(roomId: String): Flowable<AmityRoom> = RoomLiveObjectUseCase().execute(roomId)
    /* end_public_function */

    fun fetchRoom(roomId: String): Completable = FetchRoomUseCase().execute(roomId)

    /* begin_public_function
      id: room.create
      api_style: async
    */
    fun createRoom(
        title: String,
        description: String? = null,
        thumbnailFileId: String? = null,
        metadata: JsonObject? = null,
        liveChatEnabled: Boolean? = null,
        parentRoomId: String? = null,
        participants: List<String> = emptyList(),
    ): Single<AmityRoom> = CreateRoomUseCase().execute(
        title = title,
        description = description,
        thumbnailFileId = thumbnailFileId,
        metadata = metadata,
        liveChatEnabled = liveChatEnabled,
        parentRoomId = parentRoomId,
        participants = participants,
    )
    /* end_public_function */

    /* begin_public_function
      id: room.update
      api_style: async
    */
    fun updateRoom(
        roomId: String,
        title: String? = null,
        description: String? = null,
        thumbnailFileId: String? = null,
        metadata: JsonObject? = null,
        liveChatEnabled: Boolean? = null,
    ): Single<AmityRoom> = UpdateRoomUseCase().execute(
        roomId = roomId,
        title = title,
        description = description,
        thumbnailFileId = thumbnailFileId,
        metadata = metadata,
        liveChatEnabled = liveChatEnabled,
    )
    /* end_public_function */

    /* begin_public_function
      id: room.delete
      api_style: async
    */
    fun deleteRoom(roomId: String): Completable = DeleteRoomUseCase().execute(roomId)
    /* end_public_function */

    /* begin_public_function
      id: room.stop
      api_style: async
    */
    fun stopRoom(roomId: String): Completable = StopRoomUseCase().execute(roomId)
    /* end_public_function */

    /* begin_public_function
      id: room.getBroadcasterData
      api_style: async
    */
    fun getBroadcasterData(roomId: String): Single<AmityRoomBroadcastData> =
        GetBroadcasterDataUseCase().execute(roomId)
    /* end_public_function */

    fun getRecordedUrls(roomId: String): Single<List<String>> = GetRecordedUrlsUseCase().execute(roomId)

    fun leaveRoom(roomId: String): Completable = LeaveRoomUseCase().execute(roomId)

    fun removeRoomParticipant(roomId: String, userId: String): Completable = RemoveRoomParticipantUseCase().execute(roomId, userId)

    fun getCoHostEvent(roomId: String): Flowable<AmityCoHostEvent> {
        return CoHostEventBus
            .observe()
            .filter { event: AmityCoHostEvent ->
                event.roomId == roomId
            }
            .subscribeOn(Schedulers.io())
    }
}

