package com.amity.socialcloud.sdk.social.data.event

import androidx.room.Dao
import androidx.room.Query
import com.amity.socialcloud.sdk.entity.social.event.EventEntity
import com.ekoapp.ekosdk.internal.data.dao.EkoObjectDao
import io.reactivex.rxjava3.core.Flowable

@Dao
internal abstract class EventDao : EkoObjectDao<EventEntity>() {

    @Query("SELECT * FROM event WHERE eventId = :eventId LIMIT 1")
    abstract fun getEventImpl(eventId: String): Flowable<EventEntity>
    
    fun getEvent(eventId: String): Flowable<EventEntity> {
        return getEventImpl(eventId)
    }

    @Query("SELECT * FROM event WHERE eventId = :eventId")
    abstract fun observeEventImpl(eventId: String): Flowable<List<EventEntity>>
    
    fun observeEvent(eventId: String): Flowable<List<EventEntity>> {
        return observeEventImpl(eventId)
    }

    @Query("SELECT * FROM event WHERE eventId = :eventId LIMIT 1")
    abstract fun getByIdNowImpl(eventId: String): EventEntity?

    override fun getByIdNow(id: String): EventEntity? {
        return getByIdNowImpl(id)
    }

    @Query("SELECT * FROM event WHERE eventId IN (:eventIds)")
    abstract fun getByIdsNowImpl(eventIds: List<String>): List<EventEntity>

    override fun getByIdsNow(ids: List<String>): List<EventEntity> {
        return getByIdsNowImpl(ids)
    }

    @Query("DELETE FROM event WHERE eventId = :eventId")
    abstract fun deleteByIdImpl(eventId: String): Int

    fun deleteById(eventId: String) {
        deleteByIdImpl(eventId)
    }

    @Query("DELETE FROM event")
    abstract override fun deleteAll()

    @Query("SELECT * FROM event " +
            "WHERE (:isFilterByOriginType = 0 OR originType = :originType) " +
            "AND (:isFilterByOriginId = 0 OR originId = :originId) " +
            "AND (:isFilterByUserId = 0 OR userId = :userId) " +
            "AND (:isFilterByStatus = 0 OR status = :status) " +
            "AND (:isFilterByType = 0 OR type = :type) " +
            "AND updatedAt > :now " +
            "AND eventId NOT IN " +
            "(SELECT id FROM amity_paging_id WHERE hash = :hash AND nonce = :nonce) " +
            "ORDER BY " +
            "CASE WHEN :sortBy = 'startTime' AND :orderBy = 'asc' THEN startTime END ASC, " +
            "CASE WHEN :sortBy = 'startTime' AND :orderBy = 'desc' THEN startTime END DESC, " +
            "CASE WHEN :sortBy = 'createdAt' AND :orderBy = 'asc' THEN createdAt END ASC, " +
            "CASE WHEN :sortBy = 'createdAt' AND :orderBy = 'desc' THEN createdAt END DESC " +
            "LIMIT 1")
    abstract fun getLatestEventImpl(
        isFilterByOriginType: Boolean,
        originType: String?,
        isFilterByOriginId: Boolean,
        originId: String?,
        isFilterByUserId: Boolean,
        userId: String?,
        isFilterByStatus: Boolean,
        status: String?,
        isFilterByType: Boolean,
        type: String?,
        hash: Int,
        nonce: Int,
        now: org.joda.time.DateTime,
        sortBy: String,
        orderBy: String
    ): Flowable<EventEntity>

    fun getLatestEvent(
        originType: String?,
        originId: String?,
        userId: String?,
        status: String?,
        type: String?,
        hash: Int,
        nonce: Int,
        now: org.joda.time.DateTime,
        sortBy: String,
        orderBy: String
    ): Flowable<EventEntity> {
        return getLatestEventImpl(
            isFilterByOriginType = originType != null,
            originType = originType,
            isFilterByOriginId = originId != null,
            originId = originId,
            isFilterByUserId = userId != null,
            userId = userId,
            isFilterByStatus = status != null,
            status = status,
            isFilterByType = type != null,
            type = type,
            hash = hash,
            nonce = nonce,
            now = now,
            sortBy = sortBy,
            orderBy = orderBy
        )
    }

    @Query("UPDATE event SET eventId = :eventId WHERE eventId = :eventId")
    abstract fun notifyChanges(eventId: String)
}
