package com.amity.socialcloud.sdk.model.core.invitation

import android.os.Parcelable
import com.amity.socialcloud.sdk.model.core.user.AmityUser
import com.amity.socialcloud.sdk.model.social.community.AmityCommunity
import com.amity.socialcloud.sdk.model.video.room.AmityRoom
import com.ekoapp.ekosdk.ReactorObject
import com.ekoapp.ekosdk.internal.usecase.invitation.AcceptInvitationUseCase
import com.ekoapp.ekosdk.internal.usecase.invitation.CancelInvitationUseCase
import com.ekoapp.ekosdk.internal.usecase.invitation.RejectInvitationUseCase
import io.reactivex.rxjava3.core.Completable
import kotlinx.parcelize.Parcelize
import org.joda.time.DateTime

@Parcelize
data class AmityInvitation internal constructor(
        private val invitationId: String,
        private val type: AmityInvitationType,
        private val targetId: String,
        private val targetType: String,
        internal var target: Target? = null,
        private val invitedUserId: String,
        internal var invitedUser: AmityUser? = null,
        private val inviterUserId: String,
        internal var inviterUser: AmityUser? = null,
        private val status: AmityInvitationStatus,
        private val createdAt: DateTime,
        private val updatedAt: DateTime,
) : Parcelable, ReactorObject {

    sealed class Target : Parcelable  {
        @Parcelize
        data class Community(val community: AmityCommunity) : Target()

        @Parcelize
        data class Room(val roomId: String) : Target()
    }

    enum class TargetType(val value: String) {
        COMMUNITY("community"),
        ROOM("room"),
        UNKNOWN("");

        fun fromString(value: String): TargetType {
            return when (value) {
                COMMUNITY.value -> COMMUNITY
                ROOM.value -> ROOM
                else -> UNKNOWN
            }
        }
    }

    fun getInvitationId(): String {
        return invitationId
    }

    fun getType(): AmityInvitationType {
        return type
    }

    fun getTargetId(): String {
        return targetId
    }

    fun getTargetType(): String {
        return targetType
    }

    fun getTarget(): Target? {
        return target
    }

    fun getInvitedUserId(): String {
        return invitedUserId
    }

    fun getInvitedUser(): AmityUser? {
        return invitedUser
    }

    fun getStatus(): AmityInvitationStatus {
        return status
    }

    fun getInviterUserId(): String {
        return inviterUserId
    }

    fun getInviterUser(): AmityUser? {
        return inviterUser
    }

    fun getCreatedAt(): DateTime {
        return createdAt
    }

    override fun updatedAt(): DateTime {
        return createdAt
    }

    override fun uniqueId(): String {
        return invitationId
    }

    fun accept(): Completable {
        return AcceptInvitationUseCase().execute(invitationId)
    }

    fun reject(): Completable {
        return RejectInvitationUseCase().execute(invitationId)
    }

    fun cancel(): Completable {
        return CancelInvitationUseCase().execute(invitationId)
    }

}