package com.amity.socialcloud.sdk.core

import com.amity.socialcloud.sdk.core.domain.reaction.SyncLiveReactionsUseCase
import com.amity.socialcloud.sdk.core.domain.reaction.SyncRoomReactionsUseCase
import com.amity.socialcloud.sdk.core.session.component.SessionComponent
import com.amity.socialcloud.sdk.core.session.eventbus.NetworkConnectionEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.core.session.model.NetworkConnectionEvent
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.amity.socialcloud.sdk.model.core.reaction.live.AmityLiveReaction
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.disposables.CompositeDisposable
import io.reactivex.rxjava3.schedulers.Schedulers
import java.util.concurrent.TimeUnit

internal class LiveReactionSyncEngine(
    sessionLifeCycleEventBus: SessionLifeCycleEventBus,
    sessionStateEventBus: SessionStateEventBus
) : SessionComponent(sessionLifeCycleEventBus, sessionStateEventBus) {

    private var isActive: Boolean = false
    private val buffer: MutableMap<ReactionDomainId,MutableList<AmityLiveReaction>> = mutableMapOf()
    private val timer = Flowable.interval(SYNC_INTERVAL_MS, TimeUnit.MILLISECONDS)
        .flatMapCompletable {
            syncLiveReactions()
                .onErrorComplete()
        }
        .subscribeOn(Schedulers.io())

    private val disposable = CompositeDisposable()
    private var isSyncing = false

    init {
        if (sessionStateEventBus.getCurrentEvent() == SessionState.Established) {
            startReactionsSync()
        }

        NetworkConnectionEventBus
            .observe()
            .doOnNext { event ->
                when (event) {
                    is NetworkConnectionEvent.Connected -> {
                        startReactionsSync()
                    }
                    else -> {
                        stopReactionsSync()
                    }
                }
            }
            .subscribeOn(Schedulers.io())
            .subscribe()
            .let(disposable::add)
    }

    private fun syncLiveReactions(): Completable {
        return if (buffer.isEmpty() || !isActive || isSyncing) {
            Completable.complete()
        } else {
            isSyncing = true
            val domainId = buffer.keys.find {
                buffer[it] != null && buffer[it]!!.isNotEmpty()
            } ?: return Completable.complete()
            val reactions = getReactionsToSync(domainId)

            if (reactions.isNotEmpty()) {
                sendLiveReactions(domainId, reactions)
            } else {
                isSyncing = false
                Completable.complete()
            }
        }
    }

    private fun sendLiveReactions(domainId: ReactionDomainId, reactions: List<AmityLiveReaction>): Completable {
        // Clear buffer
        clearBuffer(domainId)
        return when (domainId.type) {
            ReactionDomainType.LIVESTREAM -> {
                SyncLiveReactionsUseCase().execute(domainId.id, reactions)
                    .doOnComplete {
                        isSyncing = false
                    }
                    .doOnError {
                        isSyncing = false
                    }
            }
            ReactionDomainType.ROOM -> {
                SyncRoomReactionsUseCase().execute(domainId.id, reactions)
                    .doOnComplete {
                        isSyncing = false
                    }
                    .doOnError {
                        isSyncing = false
                    }
            }
            else -> {
                Completable.complete()
            }
        }
    }

    private fun getReactionsToSync(domainId: ReactionDomainId): List<AmityLiveReaction> {
        return synchronized(buffer) {
            buffer[domainId]  ?: emptyList()
        }
    }

    private fun clearBuffer(domainId: ReactionDomainId) {
        synchronized(buffer) {
            buffer.remove(domainId)
            buffer
                .filter {
                    it.value == null || it.value.isEmpty()
                }
                .map {
                    buffer.remove(it.key)
                }
        }
    }

    private fun startReactionsSync() {
        isActive = true
        if (disposable.size() == 0 || disposable.isDisposed) {
            disposable.clear()
            timer.subscribe().let(disposable::add)
        }
    }

    private fun stopReactionsSync() {
        isActive = false
        if (disposable.size() > 0) {
            disposable.clear()
        }
        isSyncing = false
    }

    fun createReaction(domainId: ReactionDomainId, liveReaction: AmityLiveReaction) {
        synchronized(buffer) {
            if (!buffer.containsKey(domainId)) {
                buffer[domainId] = mutableListOf()
            }
            buffer[domainId]?.add(liveReaction)
        }
    }

    override fun onSessionStateChange(sessionState: SessionState) {
        when (sessionState) {
            SessionState.Established -> {
                startReactionsSync()
            }
            else -> {
                stopReactionsSync()
            }
        }
    }

    override fun establish(account: EkoAccount) {
        startReactionsSync()
    }

    override fun destroy() {
        stopReactionsSync()
    }

    override fun handleTokenExpire() {
        stopReactionsSync()
    }

    companion object {
        // Sync interval in milliseconds
        val SYNC_INTERVAL_MS = 1000L

        // Max buffer size for live reactions
        val BUFFER_MAX_SIZE = 100
    }

    data class ReactionDomainId(val id: String, val type: ReactionDomainType)

    enum class ReactionDomainType {
        LIVESTREAM, ROOM
    }
}