package com.amity.socialcloud.sdk.social.data.event.paging

import androidx.paging.ExperimentalPagingApi
import co.amity.rxbridge.toRx2
import com.amity.socialcloud.sdk.entity.social.event.EventResponseEntity
import com.amity.socialcloud.sdk.model.social.event.AmityEventResponse
import com.amity.socialcloud.sdk.model.social.event.AmityEventResponseStatus
import com.amity.socialcloud.sdk.social.data.event.AmityEventResponseRepository
import com.amity.socialcloud.sdk.social.data.event.EventRemoteDataStore
import com.amity.socialcloud.sdk.social.data.event.EventResponsePersister
import com.ekoapp.ekosdk.internal.api.dto.EventRSVPListDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single

@ExperimentalPagingApi
internal class EventResponseQueryMediator(
    private val eventId: String,
    private val status: AmityEventResponseStatus?
) : DynamicQueryStreamMediator<EventResponseEntity, EventRSVPListDto, AmityEventResponse>(
    nonce = AmityNonce.EVENT_RSVP_QUERY,
    dynamicQueryStreamKeyCreator = EventResponseQueryKeyCreator(
        eventId = eventId,
        status = status
    )
) {

    override fun forceRefresh() = true

    override fun provideReactorPublisher(): Flowable<AmityEventResponse> {
        return AmityEventResponseRepository().getLatestEventResponse(
            eventId = eventId,
            status = status,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = nonce
        ).toRx2()
    }

    override fun getFirstPageRequest(pageSize: Int): Single<EventRSVPListDto> {
        return getRequest(limit = pageSize)
    }

    override fun getFetchByTokenRequest(token: String): Single<EventRSVPListDto> {
        return getRequest(token = token)
    }

    override fun persistResponse(dto: EventRSVPListDto): Completable {
        return EventResponsePersister().persist(dto).toRx2()
    }

    override fun convertResponseToQueryToken(dto: EventRSVPListDto): DynamicQueryStreamQueryToken {
        return DynamicQueryStreamQueryToken(
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            next = dto.paging?.next,
            previous = dto.paging?.previous,
            primaryKeys = dto.eventResponses?.mapNotNull { "${it.eventId}_${it.userId}" } ?: emptyList()
        )
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<EventRSVPListDto> {
        return EventRemoteDataStore().queryRSVPs(
            eventId = eventId,
            status = status,
            limit = limit,
            token = token
        ).toRx2()
    }
}
