package com.amity.socialcloud.sdk.social.data.comment

import androidx.paging.PagingSource
import com.amity.socialcloud.sdk.api.social.comment.query.AmityCommentDataTypeFilter
import com.amity.socialcloud.sdk.api.social.comment.query.AmityCommentSortOption
import com.amity.socialcloud.sdk.model.social.comment.AmityComment
import com.ekoapp.ekosdk.internal.PostEntity
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.entity.CommentEntity
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.schedulers.Schedulers

internal class CommentLocalDataStore {

    fun saveComments(comments: List<CommentEntity>): Completable {
        return Completable.fromAction {
            UserDatabase.get().commentDao().save(comments)
        }
    }

    fun getComment(commentId: String): CommentEntity? {
        var comment: CommentEntity? = null
        Completable.fromCallable {
            val entity = UserDatabase.get().commentDao().getByIdNow(commentId)
            if (entity != null) {
                comment = entity
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return comment
    }

    fun notifyParentComment(parentId: String) {
        UserDatabase.get().commentDao().updateComment(parentId)
    }

    fun getLatestComments(
        referenceType: String,
        referenceId: String
    ): List<AmityComment> {
        var comments = emptyList<AmityComment>()
        Completable.fromCallable {
            val commentDao = UserDatabase.get().commentDao()
            val entities = commentDao.getLatestComments(referenceType, referenceId)
            if (entities.isNotEmpty()) {
                comments = entities.map {
                    CommentModelMapper().map(it)
                }
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return comments
    }

    fun getLatestReplies(
        referenceType: String,
        referenceId: String,
        parentId: String
    ): List<AmityComment> {
        var comments = emptyList<AmityComment>()
        Completable.fromCallable {
            val commentDao = UserDatabase.get().commentDao()
            val entities = commentDao.getLatestReplies(
                parentId,
                referenceType,
                referenceId,
                AmityCommentSortOption.LAST_CREATED
            )
            if (entities.isNotEmpty()) {
                comments = entities.map {
                    CommentModelMapper().map(it)
                }
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return comments
    }

    fun observeComment(commentId: String): Flowable<CommentEntity> {
        return UserDatabase.get().commentDao().getById(commentId)
    }

    fun markDeletedAfterCommentId(commentId: String) {
        UserDatabase.get().commentDao().markAllDeletedAfterCommentId(commentId)
    }

    fun markDeletedBeforeCommentId(commentId: String) {
        UserDatabase.get().commentDao().markAllDeletedBeforeCommentId(commentId)
    }

    fun hardDelete(commentId: String): Completable {
        return UserDatabase.get().commentDao().deleteById(commentId)
    }

    fun softDelete(commentId: String): Completable {
        return UserDatabase.get().commentDao().softDeleteById(commentId)
    }

    fun hasInLocal(commentId: String): Boolean {
        var hasInLocal = false
        Completable.fromCallable {
            val commentDao = UserDatabase.get().commentDao()
            val entity = commentDao.getByIdNow(commentId)
            if (entity != null) {
                hasInLocal = true
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return hasInLocal
    }

    fun deleteByReferenceId(referenceId: String): Completable {
        val commentDao = UserDatabase.get().commentDao()
        return commentDao.deleteByReferenceId(referenceId)
    }

    fun getCommentPagingSource(
        referenceId: String,
        referenceType: String,
        isFilterByParentId: Boolean,
        parentId: String?,
        isDeleted: Boolean?,
        sortOption: AmityCommentSortOption,
        dataTypeFilter: AmityCommentDataTypeFilter?
    ): PagingSource<Int, CommentEntity> {
        return UserDatabase.get().commentPagingDao().getCommentPagingSource(
            referenceId = referenceId,
            referenceType = referenceType,
            isFilterByParentId = isFilterByParentId,
            parentId = parentId,
            isDeleted = isDeleted,
            sortOption = sortOption,
            dataTypeFilter = dataTypeFilter
        )
    }

    fun getLatestComment(
        referenceId: String,
        referenceType: String,
        parentId: String?,
        isDeleted: Boolean?,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ): Flowable<CommentEntity> {
        return UserDatabase.get().commentDao().getLatestComment(
            referenceType,
            referenceId,
            parentId,
            isDeleted,
            dynamicQueryStreamKeyCreator.hashCode(),
            nonce
        )
    }

    fun getCommentByIds(postIds: List<String>) : Flowable<List<CommentEntity>> {
        return UserDatabase.get().commentDao().getByCommentIds(postIds)
    }

    fun decrementChildCount(parentId: String) {
        UserDatabase.get().commentDao().decrementChildCount(parentId)
    }

}