package com.amity.socialcloud.sdk.core.domain.session

import com.amity.socialcloud.sdk.core.CoreClient
import com.amity.socialcloud.sdk.core.LogoutProcessState
import com.amity.socialcloud.sdk.core.data.session.SessionRepository
import com.amity.socialcloud.sdk.core.session.eventbus.AppEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.model.AppEvent
import com.amity.socialcloud.sdk.core.session.model.SessionLifeCycle
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers
import org.joda.time.DateTime
import java.util.concurrent.TimeUnit

internal class LoginUseCase {

    fun execute(
        appEventBus: AppEventBus?,
        sessionLifeCycleEventBus: SessionLifeCycleEventBus?,
        userId: String?,
        displayName: String?,
        authToken: String?,
        authSignature: String?,
        authSignatureExpiresAt: DateTime?,
        isLegacyVersion: Boolean,
    ): Single<EkoAccount> {
        val loginCompletable = if (userId != null) {
            verifyUserId(
                appEventBus = appEventBus,
                userId = userId
            )
                .andThen(Completable.defer { waitForLogoutCompletion() })
                .andThen(
                    Completable.defer {
                        appEventBus?.publish(AppEvent.LoggingIn)
                    SessionRepository().login(
                        userId = userId,
                        displayName = displayName,
                        authToken = authToken,
                        isLegacyVersion = isLegacyVersion
                    )
                }
            )
        } else {
            Completable.defer {
                appEventBus?.publish(AppEvent.LoggingIn)
                SessionRepository().loginVisitor(
                    displayName = displayName,
                    authSignature = authSignature,
                    authSignatureExpiresAt = authSignatureExpiresAt,
                )
            }
        }

        return handleLoggingIn(
            appEventBus = appEventBus,
            sessionLifeCycleEventBus = sessionLifeCycleEventBus,
            loginCompletable = loginCompletable
        )
    }

    private fun handleLoggingIn(
            appEventBus: AppEventBus?,
            sessionLifeCycleEventBus: SessionLifeCycleEventBus?,
            loginCompletable: Completable,
    ): Single<EkoAccount> {
        return loginCompletable.andThen(
                Single.defer { SessionRepository().getCurrentAccount() }
        ).doOnSuccess {
            appEventBus?.publish(AppEvent.LoginSuccess)
            sessionLifeCycleEventBus?.publish(SessionLifeCycle.Establish(it))
        }.doOnError {
            val amityError = AmityException.fromThrowable(it)
            if (amityError.code == AmityError.USER_IS_GLOBAL_BANNED.code ||
                    amityError.code == AmityError.UNAUTHORIZED_ERROR.code
            ) {
                appEventBus?.publish(AppEvent.TerminationCodeReceive(amityError))
            } else {
                appEventBus?.publish(AppEvent.LoginFail)
            }
        }
    }

    private fun waitForLogoutCompletion(): Completable {
        return Flowable.interval(100, TimeUnit.MILLISECONDS)
            .filter { CoreClient.logoutProcessState != LogoutProcessState.LOGGING_OUT }
            .firstElement() // Stop as soon as the condition is met
            .ignoreElement() // Convert to Completable
            .subscribeOn(Schedulers.io()) // Ensure the polling happens on a background thread
    }

    private fun verifyUserId(
        appEventBus: AppEventBus?,
        userId: String,
    ): Completable {
        return Completable.fromCallable {
            val currentUserId = ActiveUserIdGetUseCase().execute()
            if (currentUserId.isNotEmpty() && userId != currentUserId) {
                appEventBus?.publish(AppEvent.ManualLogout)
            }
        }
    }
}