package com.amity.socialcloud.sdk.api.social.post.query

import androidx.paging.PagingData
import com.amity.socialcloud.sdk.api.social.post.review.AmityReviewStatus
import com.amity.socialcloud.sdk.model.social.feed.AmityFeedType
import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.ekoapp.ekosdk.internal.usecase.post.GetCommunityPostPagingDataUseCase
import io.reactivex.rxjava3.core.Flowable

class AmityCommunityFeedQuery private constructor(
    private val communityId: String,
    private val sortOption: AmityCommunityFeedSortOption,
    private val isDeleted: Boolean?,
    private val feedType: AmityFeedType,
    private val dataTypes: List<AmityPost.DataType>,
    private val matchingOnlyParentPosts: Boolean?,
    private val includeMixedStructure: Boolean
) {

    /**
     * Queries for Community feed.
     * @return Flowable<PagingData<AmityPost>>.
     */
    fun query(): Flowable<PagingData<AmityPost>> {
        return GetCommunityPostPagingDataUseCase().execute(
            communityId = communityId,
            sortOption = sortOption,
            isDeleted = isDeleted,
            feedType = feedType,
            dataTypes = dataTypes,
            matchingOnlyParentPosts =  matchingOnlyParentPosts ?: dataTypes.isEmpty() || dataTypes.contains(AmityPost.DataType.TEXT),
            includeMixedStructure = includeMixedStructure
        )
    }

    class Builder internal constructor(private val communityId: String) {

        private var sortOption: AmityCommunityFeedSortOption =
            AmityCommunityFeedSortOption.LAST_CREATED
        private var isDeleted: Boolean? = null
        private var feedType: AmityFeedType = AmityFeedType.PUBLISHED
        private var dataTypes: List<AmityPost.DataType> = listOf(
            AmityPost.DataType.TEXT,
            AmityPost.DataType.IMAGE,
            AmityPost.DataType.VIDEO,
            AmityPost.DataType.POLL,
            AmityPost.DataType.LIVE_STREAM,
            AmityPost.DataType.FILE,
            AmityPost.DataType.AUDIO,
            AmityPost.DataType.CLIP
        )
        private var includeMixedStructure: Boolean = false
        private var matchingOnlyParentPosts: Boolean? = null
        /**
         * Specifies [sortOption] param.
         * When [sortOption] is AmityCommunityFeedSortOption.FIRST_CREATED, posts will be sorted by createdAt property ASC.
         * When [sortOption] is AmityCommunityFeedSortOption.LAST_CREATED, posts will be sorted by createdAt property DESC.
         * The default value of [sortOption] is AmityCommunityFeedSortOption.LAST_CREATED.
         * @return the same AmityCommunityFeedQuery.Builder instance.
         */
        fun sortBy(sortOption: AmityCommunityFeedSortOption): Builder {
            return apply { this.sortOption = sortOption }
        }

        /**
         * Specifies [includeDeleted] param.
         * When [includeDeleted] is true, the result will include soft-deleted posts.
         * When [includeDeleted] is false, the result will exclude soft-deleted posts.
         * The default value of [includeDeleted] is true.
         * @return the same AmityCommunityFeedQuery.Builder instance.
         */
        fun includeDeleted(includeDeleted: Boolean): Builder {
            return apply {
                this.isDeleted = if(includeDeleted) null else false
            }
        }

        @Deprecated("Use reviewStatus() instead")
        fun feedType(feedType: AmityFeedType): Builder {
            return apply { this.feedType = feedType }
        }

        fun dataTypes(dataTypes: List<AmityPost.DataType>): Builder {
            return apply {
                this.dataTypes = dataTypes
            }
        }

        /**
         * Specifies [reviewStatus] param.
         * When [reviewStatus] is AmityReviewStatus.PUBLISHED, the results will contain only posts with AmityReviewStatus.PUBLISHED reviewStatus.
         * When [reviewStatus] is AmityReviewStatus.UNDER_REVIEW, the results will contain only posts with AmityReviewStatus.UNDER_REVIEW reviewStatus.
         * When [reviewStatus] is AmityReviewStatus.DECLINED, the results will contain only posts with AmityReviewStatus.DECLINED reviewStatus.
         * The default value of [reviewStatus] is AmityReviewStatus.PUBLISHED.
         * @return the same AmityCommunityFeedQuery.Builder instance.
         */
        fun reviewStatus(reviewStatus: AmityReviewStatus): Builder {
            return apply { this.feedType = AmityFeedType.enumOf(reviewStatus.apiKey) }
        }

        fun matchingOnlyParentPosts(matchingOnlyParentPosts: Boolean): Builder {
            return apply {
                this.matchingOnlyParentPosts = matchingOnlyParentPosts
            }
        }

        fun includeMixedStructure(includeMixedStructure: Boolean): Builder {
            return apply { this.includeMixedStructure = includeMixedStructure }
        }

        /**
         * Instantiates AmityCommunityFeedQuery with built params.
         * @return new instance of AmityCommunityFeedQuery.
         */
        fun build(): AmityCommunityFeedQuery {
            return AmityCommunityFeedQuery(
                communityId = communityId,
                sortOption = sortOption,
                isDeleted = isDeleted,
                feedType = feedType,
                dataTypes = dataTypes,
                matchingOnlyParentPosts = matchingOnlyParentPosts,
                includeMixedStructure = includeMixedStructure
            )
        }
    }

}