package com.amity.socialcloud.sdk.social.data.event.paging

import androidx.paging.PagingState
import androidx.paging.rxjava3.RxPagingSource
import com.amity.socialcloud.sdk.api.social.event.query.AmityEventOrderOption
import com.amity.socialcloud.sdk.api.social.event.query.AmityEventSortOption
import com.amity.socialcloud.sdk.entity.social.event.EventEntity
import com.amity.socialcloud.sdk.model.social.event.AmityEventOriginType
import com.amity.socialcloud.sdk.model.social.event.AmityEventStatus
import com.amity.socialcloud.sdk.model.social.event.AmityEventType
import com.amity.socialcloud.sdk.social.data.event.EventEntityMapper
import com.amity.socialcloud.sdk.social.data.event.EventQueryPersister
import com.amity.socialcloud.sdk.social.data.event.EventRemoteDataStore
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers

internal class EventQueryPagingSource(
    private val originType: AmityEventOriginType?,
    private val originId: String?,
    private val userId: String?,
    private val status: AmityEventStatus?,
    private val type: AmityEventType?,
    private val onlyAttendee: Boolean?,
    private val sortBy: AmityEventSortOption = AmityEventSortOption.START_TIME,
    private val orderBy: AmityEventOrderOption = AmityEventOrderOption.ASCENDING
) : RxPagingSource<String, EventEntity>() {

    override fun loadSingle(params: LoadParams<String>): Single<LoadResult<String, EventEntity>> {
        return EventRemoteDataStore().queryEvents(
            originType = originType,
            originId = originId,
            userId = userId,
            status = status,
            type = type,
            onlyAttendee = onlyAttendee,
            sortBy = sortBy,
            orderBy = orderBy,
            limit = params.loadSize,
            token = params.key
        )
            .flatMap { dto ->
                EventQueryPersister().persist(dto)
                    .andThen(
                        Single.just(
                            LoadResult.Page(
                                data = dto.events?.mapNotNull { eventDto ->
                                    EventEntityMapper().map(listOf(eventDto)).firstOrNull()
                                } ?: emptyList(),
                                prevKey = null,
                                nextKey = dto.paging?.next
                            ) as LoadResult<String, EventEntity>
                        )
                    )
            }
            .onErrorReturn { error ->
                LoadResult.Error(error)
            }
            .subscribeOn(Schedulers.io())
    }

    override fun getRefreshKey(state: PagingState<String, EventEntity>): String? {
        return null
    }
}
