package com.amity.socialcloud.sdk.video.domain.room.compose

import com.amity.socialcloud.sdk.core.domain.user.UserGetUseCase
import com.amity.socialcloud.sdk.chat.domain.channel.GetChannelUseCase
import com.amity.socialcloud.sdk.core.data.room.RoomRepository
import com.amity.socialcloud.sdk.core.domain.user.UserGetByIdsNowUseCase
import com.amity.socialcloud.sdk.core.domain.user.UserGetByIdsUseCase
import com.amity.socialcloud.sdk.model.video.room.AmityRoom
import com.amity.socialcloud.sdk.model.video.stream.AmityStream
import com.amity.socialcloud.sdk.model.video.stream.AmityStream.ReferenceType.Companion.enumOf
import com.amity.socialcloud.sdk.social.domain.post.PostGetUseCase
import com.amity.socialcloud.sdk.video.domain.room.GetRoomByIdsUseCase
import com.amity.socialcloud.sdk.video.domain.stream.GetStreamByIdsUseCase

internal class RoomComposerUseCase {

    private val roomRepository = RoomRepository()

    fun execute(room: AmityRoom): AmityRoom {
        addCreator(room)
        addChannel(room)
        addModeration(room)
        addReference(room)
        addChildRooms(room)
        addParticipants(room)
        return room
    }

    private fun addCreator(room: AmityRoom) {
        room.getCreatorId()?.let { id ->
            room.creator = UserGetUseCase().execute(id)
        }
    }

    private fun addChannel(room: AmityRoom) {
        room.getChannelId()?.let { channelId ->
            room.channel = GetChannelUseCase().execute(channelId)
        }
    }

    private fun addModeration(room: AmityRoom) {
        room.moderation = roomRepository.getRoomModerationByRoom(room.getRoomId())
    }

    private fun addReference(room: AmityRoom) {
        val id = room.getReferenceId()
        val type = room.getReferenceType()?.let(AmityRoom.ReferenceType::enumOf)
        if (type == AmityRoom.ReferenceType.POST && id != null) {
            room.post = PostGetUseCase().execute(id)
        }
    }

    private fun addChildRooms(parentRoom: AmityRoom) {
        if(parentRoom.isParent() && parentRoom.getChildRoomIds().isNotEmpty()) {
            parentRoom.childRooms = GetRoomByIdsUseCase().execute(parentRoom.getChildRoomIds())
        }
    }

    private fun addParticipants(room: AmityRoom) {
        room.getParticipants()
            .map { participant ->
                participant.userId.let { userId ->
                    UserGetUseCase().execute(userId)
                }
                    ?.let { user ->
                        AmityRoom.Participant(
                            userId = participant.userId,
                            type = participant.type,
                            user = user
                        )
                    }
                    ?: participant
            }
            .let { participants ->
                room.participants = participants
            }
    }
}

