package com.amity.socialcloud.sdk.social.data.event

import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.amity.socialcloud.sdk.model.social.event.AmityEventOriginType
import com.amity.socialcloud.sdk.model.social.event.AmityEventStatus
import com.amity.socialcloud.sdk.model.social.event.AmityEventType
import com.ekoapp.ekosdk.internal.api.AmityHttpClient
import com.ekoapp.ekosdk.internal.api.dto.EventListDto
import com.ekoapp.ekosdk.internal.api.socket.request.EventCreateRequest
import com.ekoapp.ekosdk.internal.api.socket.request.EventUpdateRequest
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Single
import org.joda.time.DateTime

internal class EventRemoteDataStore {

    fun getEvent(eventId: String): Single<EventListDto> {
        return AmityHttpClient.get(EventApi::class).flatMap {
            it.getEvent(eventId)
        }
    }

    fun queryEvents(
        originType: AmityEventOriginType?,
        originId: String?,
        userId: String?,
        status: AmityEventStatus?,
        type: AmityEventType?,
        onlyAttended: Boolean?, // Not passed to API yet, for future implementation
        limit: Int? = null,
        token: String? = null
    ): Single<EventListDto> {
        return AmityHttpClient.get(EventApi::class).flatMap {
            it.queryEvents(
                originType = originType?.apiKey,
                originId = originId,
                userId = userId,
                status = status?.apiKey,
                type = type?.apiKey,
                limit = limit,
                token = token
            )
        }
    }

    fun getMyEvents(
        status: com.amity.socialcloud.sdk.model.social.event.AmityEventRSVPStatus?,
        limit: Int? = null,
        token: String? = null
    ): Single<EventListDto> {
        return AmityHttpClient.get(EventApi::class).flatMap {
            it.getMyEvents(
                status = status?.apiKey,
                limit = limit,
                token = token
            )
        }
    }

    fun createEvent(
        title: String?,
        description: String?,
        type: AmityEventType?,
        isInviteOnly: Boolean?,
        startTime: DateTime?,
        endTime: DateTime, // Made non-nullable (required)
        originType: AmityEventOriginType,
        originId: String,
        location: String?,
        externalUrl: String?,
        coverImageFileId: String?,
        tags: AmityTags?,
        timezone: String?
    ): Single<EventListDto> {
        val metadata = timezone?.let {
            JsonObject().apply {
                addProperty("timezone", it)
            }
        }

        val request = EventCreateRequest(
            title = title,
            description = description,
            type = type?.apiKey,
            isInviteOnly = isInviteOnly,
            startTime = startTime,
            endTime = endTime,
            originType = originType.apiKey,
            originId = originId,
            location = location,
            externalUrl = externalUrl,
            coverImageFileId = coverImageFileId,
            tags = tags?.get(),
            metadata = metadata
        )

        return AmityHttpClient.get(EventApi::class).flatMap {
            it.createEvent(request)
        }
    }

    fun updateEvent(
        eventId: String,
        title: String?,
        description: String?,
        isInviteOnly: Boolean?,
        startTime: DateTime?,
        endTime: DateTime?,
        location: String?,
        externalUrl: String?,
        coverImageFileId: String?,
        tags: AmityTags?,
        timezone: String?
    ): Single<EventListDto> {
        val metadata = timezone?.let {
            JsonObject().apply {
                addProperty("timezone", it)
            }
        }

        val request = EventUpdateRequest(
            eventId = eventId,
            title = title,
            description = description,
            isInviteOnly = isInviteOnly,
            startTime = startTime,
            endTime = endTime,
            location = location,
            externalUrl = externalUrl,
            coverImageFileId = coverImageFileId,
            tags = tags?.get(),
            metadata = metadata
        )

        return AmityHttpClient.get(EventApi::class).flatMap {
            it.updateEvent(eventId, request)
        }
    }

    fun deleteEvent(eventId: String): Single<com.ekoapp.ekosdk.internal.api.dto.EkoDeletionDto> {
        return AmityHttpClient.get(EventApi::class).flatMap {
            it.deleteEvent(eventId)
        }
    }

}
