package com.amity.socialcloud.sdk.model.video.room

import android.os.Parcelable
import com.amity.socialcloud.sdk.api.core.events.AmityTopicSubscription
import com.amity.socialcloud.sdk.core.JsonObjectParceler
import com.amity.socialcloud.sdk.model.chat.channel.AmityChannel
import com.amity.socialcloud.sdk.model.core.events.AmityRoomEvents
import com.amity.socialcloud.sdk.model.core.events.AmityTopic
import com.amity.socialcloud.sdk.model.core.file.AmityImage
import com.amity.socialcloud.sdk.model.core.invitation.AmityInvitation
import com.amity.socialcloud.sdk.model.core.user.AmityUser
import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.ekoapp.ekosdk.ReactorObject
import com.ekoapp.ekosdk.internal.usecase.invitation.CreateCohostInvitationUseCase
import com.ekoapp.ekosdk.internal.usecase.invitation.GetMyCohostInvitationUseCase
import com.ekoapp.ekosdk.internal.usecase.invitation.ObserveMyCohostInvitationUseCase
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import kotlinx.parcelize.Parcelize
import kotlinx.parcelize.TypeParceler
import org.joda.time.DateTime

@Parcelize
@TypeParceler<JsonObject?, JsonObjectParceler>
data class AmityRoom internal constructor(
        private val roomId: String,
        private val type: AmityRoomType,
        private val targetType: String?,
        private val targetId: String?,
        private val referenceType: String?,
        private val referenceId: String?,
        private val channelEnabled: Boolean,
        private val channelId: String?,
        private val title: String?,
        private val description: String?,
        private val thumbnailFileId: String?,
        private val status: AmityRoomStatus,
        private val livePlaybackUrl: String?,
        private val durationSeconds: Int?,
        private val parentRoomId: String?,
        private val childRoomIds: List<String>,
        internal var childRooms: List<AmityRoom>,
        private val creatorId: String?,
        private val isDeleted: Boolean,
        private val metadataJson: JsonObject?,
        private val createdAt: DateTime?,
        private val updatedAt: DateTime?,
        private val liveAt: DateTime?,
        private val endedAt: DateTime?,
        private val recordedAt: DateTime?,
        private val recordedUrl: List<String?>?,
        private val recordedThumbnailUrl: List<String?>?,
        internal val path: String,
        internal var post: AmityPost?,
        internal var creator: AmityUser? = null,
        internal var channel: AmityChannel? = null,
        internal var participants: List<Participant> = emptyList(),
        internal var thumbnail: AmityImage? = null,
) : Parcelable, ReactorObject {

    internal var moderation: AmityRoomModeration? = null

    fun getRoomId() = roomId
    fun getStatus() = status
    fun getTitle() = title
    fun getDescription() = description
    fun isChannelEnabled() = channelEnabled
    fun getChannelId() = channelId
    fun getChannel() = channel
    fun getReferenceId() = referenceId
    fun getReferenceType() = referenceType
    fun getParticipants() = participants
    fun getChildRoomIds() = childRoomIds
    fun getChildRooms() = childRooms
    fun getParentRoomId() = parentRoomId
    fun isParent() = parentRoomId.isNullOrBlank()
    fun getCreatorId() = creatorId
    fun getCreator() = creator
    fun isDeleted() = isDeleted
    fun getCreatedAt() = createdAt
    fun getUpdatedAt() = updatedAt
    fun getLiveAt() = liveAt
    fun getEndedAt() = endedAt
    fun getRecordedAt() = recordedAt
    fun getLivePlaybackUrl() = livePlaybackUrl
    fun getMetadata() = metadataJson
    fun getModeration(): AmityRoomModeration? = moderation
    fun getThumbnailFileId() = thumbnailFileId
    fun getThumbnail() = thumbnail

    fun getRecordedPlaybackInfos(): List<RecordedPlaybackInfo> {
        return recordedUrl?.mapIndexed { index, url ->
            RecordedPlaybackInfo(
                    url = url,
                    thumbnailUrl = recordedThumbnailUrl?.getOrNull(index)
            )
        } ?: emptyList()
    }

    override fun updatedAt(): DateTime? {
        return updatedAt
    }

    override fun uniqueId(): String {
        return roomId
    }

    fun subscription(events: AmityRoomEvents): AmityTopicSubscription {
        return AmityTopicSubscription(AmityTopic.ROOM(this, events))
    }

    fun getPost(): AmityPost? {
        return post
    }

    fun createInvitation(userId: String): Completable {
        return CreateCohostInvitationUseCase().execute(roomId, userId)
    }

    fun getInvitation(): Single<List<AmityInvitation>> {
        return GetMyCohostInvitationUseCase().execute(roomId)
    }

    fun observeInvitation(): Flowable<AmityInvitation> {
        return ObserveMyCohostInvitationUseCase().execute(roomId)
    }

    enum class ReferenceType(val apiKey: String) {
        POST("post");

        companion object {
            fun enumOf(value: String?): ReferenceType {
                return values().find { it.apiKey == value } ?: POST
            }
        }
    }

    data class RecordedPlaybackInfo(
            var url: String? = null,
            var thumbnailUrl: String? = null,
    )

    @Parcelize
    class Participant(
            val type: ParticipantType,
            val userId: String,
            val user: AmityUser? = null,
    ) : Parcelable

    enum class ParticipantType(val apiKey: String) {
        HOST("host"), CoHost("coHost"), Unknown("unknown");

        companion object {
            fun enumOf(value: String?): ParticipantType {
                return values().find { it.apiKey == value } ?: Unknown
            }
        }
    }
}

enum class AmityRoomType(val apiKey: String) { CO_HOSTS("coHosts"), DIRECT_STREAMING("directStreaming"); companion object { fun enumOf(v:String?)= values().find{it.apiKey==v}?:CO_HOSTS } }

enum class AmityRoomStatus(val apiKey: String) { IDLE("idle"), LIVE("live"), WAITING_RECONNECT("waitingReconnect"), ENDED("ended"), RECORDED("recorded"); companion object { fun enumOf(v:String?)= values().find{it.apiKey==v}?:IDLE } }

enum class AmityRoomSortOption(val apiKey: String) { FirstCreated("firstCreated"), LastCreated("lastCreated"); }
