package com.amity.socialcloud.sdk.core.data.user

import androidx.paging.PagingSource
import com.amity.socialcloud.sdk.api.core.user.search.AmityUserSearchMatchType
import com.amity.socialcloud.sdk.api.core.user.search.AmityUserSortOption
import com.amity.socialcloud.sdk.entity.core.user.UserEntity
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.entity.UserPresenceEntity
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.schedulers.Schedulers

internal class UserLocalDataStore {

    fun saveUsers(users: List<UserEntity>): Completable {
        return Completable.fromAction { UserDatabase.get().userDao().save(users) }
    }

    fun saveUserPresence(presences: List<UserPresenceEntity>): Completable {
        return Completable.fromAction {
            val userDao = UserDatabase.get().userDao()

            val entities = presences.mapNotNull { presence ->
                val entity = userDao.getByIdNow(presence.userId)
                entity?.lastHeartbeat = presence.lastHeartbeat
                entity
            }

            userDao.save(entities)
        }
    }

    fun getUser(userId: String): UserEntity? {
        var user: UserEntity? = null
        Completable.fromCallable {
            val entity = UserDatabase.get().userDao().getByIdNow(userId)
            if (entity != null) {
                user = entity
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return user
    }

    fun getUserNow(userId: String): UserEntity? {
        var user: UserEntity? = null
        val entity = UserDatabase.get().userDao().getByIdNow(userId)
        if (entity != null) {
            user = entity
        }
        return user
    }

    fun getUserByIdsNow(userIds: List<String>): List<UserEntity> {
        return UserDatabase.get().userDao().getByIdsNow(userIds)
    }

    fun getUserByIds(userIds: List<String>): Flowable<List<UserEntity>> {
        return UserDatabase.get().userDao().getByUserIds(userIds)
    }

    fun observeUser(userId: String): Flowable<UserEntity> {
        return UserDatabase.get().userDao().observeUser(userId)
            .map {
                if (it.isEmpty()) {
                    throw AmityException.create(
                        message = "item not found",
                        cause = null,
                        error = AmityError.ITEM_NOT_FOUND
                    )
                } else {
                    return@map it[0]
                }
            }
    }

    fun getUserPagingSource(
        keyword: String?,
        matchType: AmityUserSearchMatchType?,
        sortBy: AmityUserSortOption
    ): PagingSource<Int, UserEntity> {
        return UserDatabase.get().userPagingDao().getUserPagingSource(
            keyword,
            matchType,
            sortBy
        )
    }
}