package com.amity.socialcloud.sdk.api.social.event.create

import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.amity.socialcloud.sdk.model.social.event.AmityEvent
import com.amity.socialcloud.sdk.model.social.event.AmityEventType
import com.amity.socialcloud.sdk.model.social.event.AmityEventOriginType
import com.amity.socialcloud.sdk.social.domain.event.create.EventCreateUseCase
import io.reactivex.rxjava3.core.Single
import org.joda.time.DateTime

class AmityEventCreateOptions private constructor(
    private val title: String?,
    private val description: String?,
    private val type: AmityEventType?,
    private val isInviteOnly: Boolean?,
    private val startTime: DateTime?,
    private val endTime: DateTime, // Made non-nullable (required)
    private val originType: AmityEventOriginType,
    private val originId: String,
    private val location: String?,
    private val externalUrl: String?,
    private val coverImageFileId: String?,
    private val tags: AmityTags?,
    private val timezone: String?
) {

    /**
     * Creates an event
     * @return Async AmityEvent as Single<AmityEvent>.
     */
    fun create(): Single<AmityEvent> {
        return EventCreateUseCase().execute(
            title = title,
            description = description,
            type = type,
            isInviteOnly = isInviteOnly,
            startTime = startTime,
            endTime = endTime,
            originType = originType,
            originId = originId,
            location = location,
            externalUrl = externalUrl,
            coverImageFileId = coverImageFileId,
            tags = tags,
            timezone = timezone
        )
    }

    class Builder internal constructor() {
        private var title: String? = null
        private var description: String? = null
        private var type: AmityEventType? = null
        private var isInviteOnly: Boolean? = null
        private var startTime: DateTime? = null
        private var endTime: DateTime? = null // Required, must be set before build()
        private var originType: AmityEventOriginType = AmityEventOriginType.COMMUNITY // Default to COMMUNITY
        private var originId: String = "" // Default to empty string, but should be set by user
        private var location: String? = null
        private var externalUrl: String? = null
        private var coverImageFileId: String? = null
        private var tags: AmityTags? = null
        private var timezone: String? = null

        /**
         * @param [title] to set the event title
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun setTitle(title: String): Builder {
            return apply {
                this.title = title
            }
        }

        /**
         * @param [title] to set the event title
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun title(title: String): Builder = setTitle(title)

        /**
         * @param [description] to set the event description
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun setDescription(description: String): Builder {
            return apply {
                this.description = description
            }
        }

        /**
         * @param [description] to set the event description
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun description(description: String): Builder = setDescription(description)

        /**
         * @param [type] to set the event type
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun setType(type: AmityEventType): Builder {
            return apply {
                this.type = type
            }
        }

        /**
         * @param [type] to set the event type
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun type(type: AmityEventType): Builder = setType(type)

        /**
         * @param [isInviteOnly] to set whether the event is invite-only
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun setIsInviteOnly(isInviteOnly: Boolean): Builder {
            return apply {
                this.isInviteOnly = isInviteOnly
            }
        }

        /**
         * @param [isInviteOnly] to set whether the event is invite-only
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun isInviteOnly(isInviteOnly: Boolean): Builder = setIsInviteOnly(isInviteOnly)

        /**
         * @param [startTime] to set the event start time
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun setStartTime(startTime: DateTime): Builder {
            return apply {
                this.startTime = startTime
            }
        }

        /**
         * @param [startTime] to set the event start time
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun startTime(startTime: DateTime): Builder = setStartTime(startTime)

        /**
         * @param [endTime] to set the event end time
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun setEndTime(endTime: DateTime): Builder {
            return apply {
                this.endTime = endTime
            }
        }

        /**
         * @param [endTime] to set the event end time
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun endTime(endTime: DateTime): Builder = setEndTime(endTime)

        /**
         * @param [originType] to set the event origin type
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun setOriginType(originType: AmityEventOriginType): Builder {
            return apply {
                this.originType = originType
            }
        }

        /**
         * @param [originType] to set the event origin type
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun originType(originType: AmityEventOriginType): Builder = setOriginType(originType)

        /**
         * @param [originId] to set the event origin ID
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun setOriginId(originId: String): Builder {
            return apply {
                this.originId = originId
            }
        }

        /**
         * @param [originId] to set the event origin ID
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun originId(originId: String): Builder = setOriginId(originId)

        /**
         * @param [location] to set the event location
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun setLocation(location: String): Builder {
            return apply {
                this.location = location
            }
        }

        /**
         * @param [location] to set the event location
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun location(location: String): Builder = setLocation(location)

        /**
         * @param [externalUrl] to set the event external URL
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun setExternalUrl(externalUrl: String): Builder {
            return apply {
                this.externalUrl = externalUrl
            }
        }

        /**
         * @param [externalUrl] to set the event external URL
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun externalUrl(externalUrl: String): Builder = setExternalUrl(externalUrl)

        /**
         * @param [coverImageFileId] to set the event cover image file ID
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun setCoverImageFileId(coverImageFileId: String): Builder {
            return apply {
                this.coverImageFileId = coverImageFileId
            }
        }

        /**
         * @param [coverImageFileId] to set the event cover image file ID
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun coverImageFileId(coverImageFileId: String): Builder = setCoverImageFileId(coverImageFileId)

        /**
         * @param [tags] to set the event tags
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun setTags(tags: AmityTags): Builder {
            return apply {
                this.tags = tags
            }
        }

        /**
         * @param [tags] to set the event tags
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun tags(tags: AmityTags): Builder = setTags(tags)

        /**
         * @param [tags] to set the event tags from a list of strings
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun setTags(tags: List<String>): Builder {
            return apply {
                this.tags = AmityTags(tags)
            }
        }

        /**
         * @param [tags] to set the event tags from a list of strings
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun tags(tags: List<String>): Builder = setTags(tags)

        /**
         * @param [timezone] to set the event timezone (e.g., "Asia/Bangkok")
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun setTimezone(timezone: String): Builder {
            return apply {
                this.timezone = timezone
            }
        }

        /**
         * @param [timezone] to set the event timezone (e.g., "Asia/Bangkok")
         * @return the same instance of AmityEventCreateOptions.Builder
         */
        fun timezone(timezone: String): Builder = setTimezone(timezone)

        /**
         * Instantiates AmityEventCreateOptions with built params
         * @return new instance of AmityEventCreateOptions
         * @throws IllegalStateException if endTime is not set
         */
        fun build(): AmityEventCreateOptions {
            val requiredEndTime = requireNotNull(endTime) { "End time is required. Call endTime() before build()." }
            
            return AmityEventCreateOptions(
                title = title,
                description = description,
                type = type,
                isInviteOnly = isInviteOnly,
                startTime = startTime,
                endTime = requiredEndTime,
                originType = originType,
                originId = originId,
                location = location,
                externalUrl = externalUrl,
                coverImageFileId = coverImageFileId,
                tags = tags,
                timezone = timezone
            )
        }
    }
}
