package com.amity.socialcloud.sdk.api.social.community.create

import com.amity.socialcloud.sdk.model.core.file.AmityImage
import com.amity.socialcloud.sdk.model.social.community.AmityCommunity
import com.amity.socialcloud.sdk.model.social.community.AmityCommunityPostSettings
import com.amity.socialcloud.sdk.model.social.community.AmityCommunityStorySettings
import com.amity.socialcloud.sdk.social.domain.community.CreateCommunityUseCase
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Single

class AmityCommunityCreator private constructor(
    private var displayName: String,
    private var description: String? = null,
    private var isPublic: Boolean? = false,
    private var categoryIds: List<String>? = null,
    private var metadata: JsonObject? = null,
    private var userIds: List<String>? = null,
    private var avatarFileId: String? = null,
    private var needApprovalOnPostCreation: Boolean? = null,
    private var onlyAdminCanPost: Boolean? = null,
    private var storySettings: AmityCommunityStorySettings? = null,
    private var isDiscoverable: Boolean? = null,
    private var requiresJoinApproval: Boolean? = null,
) {

    /**
     * Creates a community
     * @return Async AmityCommunity as Single<AmityCommunity>.
     */
    fun create(): Single<AmityCommunity> {
        return CreateCommunityUseCase()
            .execute(
                displayName,
                description,
                categoryIds,
                isPublic,
                metadata,
                userIds,
                avatarFileId,
                needApprovalOnPostCreation,
                onlyAdminCanPost,
                storySettings,
                isDiscoverable,
                requiresJoinApproval,
            )
    }

    class Builder internal constructor(
        private val displayName: String,
        private val isDiscoverable: Boolean? = null,
        private val requiresJoinApproval: Boolean? = null,
    ) {
        private var description: String? = null
        private var isPublic: Boolean? = false
        private var categoryIds: List<String>? = null
        private var metadata: JsonObject? = null
        private var userIds: List<String>? = null
        private var avatarFileId: String? = null
        private var needApprovalOnPostCreation: Boolean? = null
        private var onlyAdminCanPost: Boolean? = null
        private var storySettings: AmityCommunityStorySettings? = null

        /**
         * @param [description] the community's description value
         * @return the same instance of AmityCommunityCreator.Builder
         */
        fun description(description: String): Builder {
            return apply {
                if (description.isNotBlank()) {
                    this.description = description
                }
            }
        }

        /**
         * @param [isPublic] the community's isPublic value
         * @return the same instance of AmityCommunityCreator.Builder
         */
        fun isPublic(isPublic: Boolean): Builder {
            return apply { this.isPublic = isPublic }
        }

        /**
         * @param [categoryIds] the community's list of categoryId
         * @return the same instance of AmityCommunityCreator.Builder
         */
        fun categoryIds(categoryIds: List<String>): Builder {
            return apply { this.categoryIds = categoryIds }
        }

        /**
         * @param [metadata] the community's metadata
         * @return the same instance of AmityCommunityCreator.Builder
         */
        fun metadata(metadata: JsonObject): Builder {
            return apply { this.metadata = metadata }
        }

        /**
         * @param [userIds] the list of userId to be added as community's member
         * @return the same instance of AmityCommunityCreator.Builder
         */
        fun userIds(userIds: List<String>): Builder {
            return apply { this.userIds = userIds }
        }

        /**
         * @param [avatar] the replacing community's avatar
         * @return the same instance of AmityCommunityCreator.Builder
         */
        fun avatar(avatar: AmityImage): Builder {
            return apply { this.avatarFileId = avatar.getFileId() }
        }

        /**
         * @param [postSettings] the community's post settings
         * @return the same instance of AmityCommunityCreator.Builder
         */
        fun postSettings(postSettings: AmityCommunityPostSettings): Builder {
            return apply {
                this.needApprovalOnPostCreation = postSettings.isPostReviewEnabled
                this.onlyAdminCanPost = postSettings.onlyAdminCanPost
            }
        }

        fun storySettings(storySettings: AmityCommunityStorySettings): Builder {
            return apply {
                this.storySettings = storySettings
            }
        }

        /**
         * Instantiates AmityCommunityCreator
         * @return new instance of AmityCommunityCreator with built params.
         */
        fun build(): AmityCommunityCreator {
            return AmityCommunityCreator(
                displayName = displayName,
                description = description,
                isPublic = isPublic,
                categoryIds = categoryIds,
                metadata = metadata,
                userIds = userIds,
                avatarFileId = avatarFileId,
                needApprovalOnPostCreation = needApprovalOnPostCreation,
                onlyAdminCanPost = onlyAdminCanPost,
                storySettings = storySettings,
                isDiscoverable = isDiscoverable,
                requiresJoinApproval = requiresJoinApproval,
            )
        }
    }

}