package com.amity.socialcloud.sdk.chat.data.channel.membership

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.api.chat.member.query.AmityChannelMembershipSortOption
import com.amity.socialcloud.sdk.chat.data.channel.membership.paging.ChannelMemberSearchMediator
import com.amity.socialcloud.sdk.chat.data.channel.membership.paging.ChannelMembershipMediator
import com.amity.socialcloud.sdk.model.chat.member.AmityChannelMember
import com.amity.socialcloud.sdk.model.chat.member.AmityMembershipType
import com.amity.socialcloud.sdk.model.core.role.AmityRoles
import com.ekoapp.ekosdk.EkoObjectRepository
import com.ekoapp.ekosdk.internal.api.dto.ChannelMembershipQueryDto
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import org.joda.time.Duration

@OptIn(ExperimentalPagingApi::class)
internal class ChannelMembershipRepository : EkoObjectRepository() {

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }

    fun observeChannelMember(channelId: String, userId: String): Flowable<AmityChannelMember> {
        return ChannelMembershipLocalDataStore().observeChannelMember(channelId, userId)
            .map {
                ChannelMembershipModelMapper().map(it)
            }
    }

    fun observeChannelMembers(channelId: String): Flowable<List<AmityChannelMember>> {
        return ChannelMembershipLocalDataStore().observeChannelMembers(channelId)
            .map { members ->
                members.map {
                    ChannelMembershipModelMapper().map(it)
                }
            }
    }

    fun addChannelMembers(channelId: String, userIds: List<String>): Completable {
        return ChannelMembershipRemoteDataStore().addChannelMembers(channelId, userIds)
            .flatMapCompletable {
                persistChannelMemberships(it)
            }
    }

    fun removeChannelMembers(channelId: String, userIds: List<String>): Completable {
        return ChannelMembershipRemoteDataStore().removeChannelMembers(channelId, userIds)
            .flatMapCompletable {
                persistChannelMemberships(it)
            }
    }

    fun getChannelMembershipPagingData(
        channelId: String,
        memberships: List<AmityMembershipType>,
        isMuted: Boolean?,
        roles: AmityRoles,
        sortOption: AmityChannelMembershipSortOption,
        isDeleted: Boolean?,
    ): Flowable<PagingData<AmityChannelMember>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = true
            ),
            dynamicQueryStreamMediator = ChannelMembershipMediator(
                channelId = channelId,
                memberships = memberships,
                isMuted = isMuted,
                roles = roles,
                sortOption = sortOption,
                isDeleted = isDeleted
            ),
            pagingSourceFactory = {
                ChannelMembershipLocalDataStore().getChannelMembershipPagingSource(
                    channelId = channelId,
                    memberships = memberships,
                    isMuted = isMuted,
                    roles = roles,
                    isDeleted = isDeleted,
                    sortOption = sortOption
                )
            },
            modelMapper = ChannelMembershipModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun searchChannelMembershipPagingData(
        channelId: String,
        memberships: List<AmityMembershipType>,
        isMuted: Boolean?,
        roles: AmityRoles,
        keyword: String,
        isDeleted: Boolean?
    ): Flowable<PagingData<AmityChannelMember>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = true
            ),
            dynamicQueryStreamMediator = ChannelMemberSearchMediator(
                channelId = channelId,
                keyword = keyword,
                memberships = memberships,
                isMuted = isMuted,
                roles = roles,
                isDeleted = isDeleted
            ),
            pagingSourceFactory = {
                ChannelMembershipLocalDataStore().getChannelMemberSearchPagingSource(
                    channelId = channelId,
                    keyword = keyword,
                    memberships = memberships,
                    isMuted = isMuted,
                    isDeleted = isDeleted,
                    roles = roles
                )
            },
            modelMapper = ChannelMembershipModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun assignChannelRole(channelId: String, role: String, userIds: List<String>): Completable {
        return ChannelMembershipRemoteDataStore().assignChannelRole(channelId, role, userIds)
            .flatMapCompletable {
                persistChannelMemberships(it)
            }
    }

    fun unassignChannelRole(channelId: String, role: String, userIds: List<String>): Completable {
        return ChannelMembershipRemoteDataStore().unassignChannelRole(channelId, role, userIds)
            .flatMapCompletable {
                persistChannelMemberships(it)
            }
    }

    fun banChannelUsers(channelId: String, userIds: List<String>): Completable {
        return ChannelMembershipRemoteDataStore().banChannelMembers(channelId, userIds)
            .flatMapCompletable {
                persistChannelMemberships(it)
            }
    }

    fun unbanChannelUsers(channelId: String, userIds: List<String>): Completable {
        return ChannelMembershipRemoteDataStore().unbanChannelMembers(channelId, userIds)
            .flatMapCompletable {
                persistChannelMemberships(it)
            }
    }

    fun muteChannelUsers(channelId: String, timeout: Duration, userIds: List<String>): Completable {
        return ChannelMembershipRemoteDataStore().muteChannelMembers(channelId, timeout, userIds)
    }

    fun unmuteChannelUsers(channelId: String, userIds: List<String>): Completable {
        return muteChannelUsers(channelId, Duration.ZERO, userIds)
    }

    fun muteChannel(channelId: String, timeout: Duration): Completable {
        return ChannelMembershipRemoteDataStore().muteChannel(channelId, timeout)
    }

    fun unmuteChannel(channelId: String): Completable {
        return muteChannel(channelId, Duration.ZERO)
    }

    fun handleMembershipBanned(channelId: String, userId: String) : Completable {
        return Completable.fromAction {
            ChannelMembershipLocalDataStore().updateMembership(channelId, userId, AmityMembershipType.BANNED)
            UserDatabase.get().channelDao().notifyChanges(channelId)
            UserDatabase.get().messageDao().softDeleteFromChannelByUserId(channelId, userId)
        }
    }

    fun handleMembershipMuted(channelId: String, userId: String, isMuted: Boolean) : Completable {
        return Completable.fromAction {
            ChannelMembershipLocalDataStore().updateMemberIsMuted(channelId, userId, isMuted)
            UserDatabase.get().channelDao().notifyChanges(channelId)
        }
    }

    fun getLatestChannel(
        channelId: String,
        roles: AmityRoles,
        memberships: List<AmityMembershipType>,
        isMuted: Boolean?,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ): Flowable<AmityChannelMember> {
        return ChannelMembershipLocalDataStore().getLatestChannel(
            channelId = channelId,
            memberships = memberships,
            isMuted = isMuted,
            roles = roles,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = nonce
        )
            .map {
                ChannelMembershipModelMapper().map(it)
            }
    }

    fun getChannelMembers(channelId: String): Single<List<AmityChannelMember>> {
        return Single.fromCallable {
            ChannelMembershipLocalDataStore().getChannelMembers(channelId)
                .map { ChannelMembershipModelMapper().map(it) }
        }
    }

    internal fun persistChannelMemberships(dto: ChannelMembershipQueryDto): Completable {
        return ChannelMembershipQueryPersister().persist(dto)
    }

}