package com.ekoapp.ekosdk.internal.data.dao

import androidx.room.Dao
import androidx.room.Query
import androidx.room.RawQuery
import androidx.room.Transaction
import androidx.sqlite.db.SimpleSQLiteQuery
import androidx.sqlite.db.SupportSQLiteQuery
import com.amity.socialcloud.sdk.api.core.AmityCoreClient.getUserId
import com.amity.socialcloud.sdk.model.chat.member.AmityMembershipType
import com.ekoapp.ekosdk.EkoChannelReadStatus
import com.ekoapp.ekosdk.EkoChannelWithMembershipAndExtra
import com.ekoapp.ekosdk.internal.EkoChannelEntity
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.dao.EkoTagDao.Companion.update
import com.ekoapp.ekosdk.internal.data.model.EkoChannelTag
import com.ekoapp.ekosdk.internal.keycreator.toSqlArray
import com.google.common.collect.FluentIterable
import io.reactivex.rxjava3.core.Flowable
import okio.Timeout
import org.joda.time.DateTime

@Dao
abstract class EkoChannelDao internal constructor() : EkoObjectDao<EkoChannelEntity>(), AmityPagingDao<EkoChannelWithMembershipAndExtra> {
	private val channelMembershipDao: EkoChannelMembershipDao
	private val messageDao: EkoMessageDao
	private val channelTagDao: EkoChannelTagDao
	private val channelExtraDao: EkoChannelExtraDao

	init {
		val db = UserDatabase.get()
		channelMembershipDao = db.channelMembershipDao()
		messageDao = db.messageDao()
		channelTagDao = db.channelTagDao()
		channelExtraDao = db.channelExtraDao()
	}

	@Query("SELECT channel.*," +
			" channel_membership.readToSegment as membership_readToSegment," +
			" channel_extra.localReadToSegment as extra_localReadToSegment," +
			" channel_membership.lastMentionedSegment as membership_lastMentionedSegment," +
			" channel_extra.localLastMentionedSegment as extra_localLastMentionedSegment" +
			" from channel, channel_membership, channel_extra" +
			" where channel.channelId = channel_membership.channelId and channel.channelId = channel_extra.channelId" +
			" and channel.channelId in (SELECT channelId from channel_membership where membership = :membership)" +
			" and channel_membership.userId = :userId")
	abstract fun getAllJoinedChannelsImpl(userId: String, membership: String): Flowable<List<EkoChannelWithMembershipAndExtra>>
	fun getAllJoinedChannels(userId: String): Flowable<List<EkoChannelWithMembershipAndExtra>> {
		return getAllJoinedChannelsImpl(userId, AmityMembershipType.MEMBER.apiKey)
	}

	@Query("SELECT *" +
			" from channel" +
			" where channel.channelType in (:types)" +
			" and case when :isFilterByMemberships then channel.channelId in (SELECT channelId from channel_membership where membership in (:memberships) and userId = :userId)" +
			" else channel.channelId is not null end" +  // always true
			" and case when :isIncludingTags then channel.channelId in (SELECT channelId from channel_tag where tagName in (:includingTags))" +
			" else channel.channelId is not null end" +  // always true
			" and case when :isExcludingTags then channel.channelId not in (SELECT channelId from channel_tag where tagName in (:excludingTags))" +
			" else channel.channelId is not null end" +  // always true
			" and case when :isDeleted is not null then channel.isDeleted = :isDeleted" +
			" else channel.channelId is not null end" +  // always true
			" and channel.updatedAt > :now" +
			" and channel.channelId not in " +
			"(" +
			"SELECT amity_paging_id.id" +
			" from amity_paging_id" +
			" where amity_paging_id.hash = (:hash)" +
			" and amity_paging_id.nonce = (:nonce) " +
			")" +
			" order by channel.updatedAt  desc" +
			" limit 1")
	abstract fun getLatestChannelImpl(types: Array<String>,
	                                  isFilterByMemberships: Boolean,
	                                  userId: String,
	                                  memberships: Array<String>,
	                                  isIncludingTags: Boolean,
	                                  includingTags: Array<String>,
	                                  isExcludingTags: Boolean,
	                                  excludingTags: Array<String>,
	                                  isDeleted: Boolean?,
	                                  hash: Int,
	                                  nonce: Int,
	                                  now: DateTime): Flowable<EkoChannelEntity>

	fun getLatestChannel(types: Array<String>,
	                     userId: String,
	                     memberships: Array<String>,
	                     includingTags: Array<String>,
	                     excludingTags: Array<String>,
	                     isDeleted: Boolean?,
	                     hash: Int,
	                     nonce: Int,
	                     now: DateTime): Flowable<EkoChannelEntity> {
		return getLatestChannelImpl(types,
				memberships.isNotEmpty(),
				userId,
				memberships,
				includingTags.isNotEmpty(),
				includingTags,
				excludingTags.isNotEmpty(),
				excludingTags,
				isDeleted,
				hash,
				nonce,
				now)
	}

	@Query("SELECT channel.*," +
			" channel_membership.readToSegment as membership_readToSegment," +
			" channel_extra.localReadToSegment as extra_localReadToSegment," +
			" channel_membership.lastMentionedSegment as membership_lastMentionedSegment," +
			" channel_extra.localLastMentionedSegment as extra_localLastMentionedSegment" +
			" from channel, channel_membership, channel_extra" +
			" where channel.channelId = channel_membership.channelId and channel.channelId = channel_extra.channelId" +
			" and channel.channelId = :channelId LIMIT 1")
	abstract fun getChannelImpl(channelId: String): Flowable<EkoChannelWithMembershipAndExtra>
	fun getChannel(channelId: String): Flowable<EkoChannelEntity> {
		return getChannelImpl(channelId).map { input: EkoChannelWithMembershipAndExtra -> input }
	}

	@Query("SELECT channel.*," +
			" channel_membership.readToSegment as membership_readToSegment," +
			" channel_extra.localReadToSegment as extra_localReadToSegment," +
			" channel_membership.lastMentionedSegment as membership_lastMentionedSegment," +
			" channel_extra.localLastMentionedSegment as extra_localLastMentionedSegment" +
			" from channel, channel_membership, channel_extra" +
			" where channel.channelId = channel_membership.channelId and channel.channelId = channel_extra.channelId" +
			" and channel.channelId = :channelId LIMIT 1")
	abstract fun getByIdNowImpl(channelId: String): EkoChannelWithMembershipAndExtra
	override fun getByIdNow(id: String): EkoChannelEntity {
		return getByIdNowImpl(id)
	}

	@Query(
		"SELECT channel.* from channel" +
				" where channel.internalChannelId = :internalId LIMIT 1"
	)
	abstract fun getByInternalIdNowImpl(internalId: String?): EkoChannelEntity?

	fun getByInternalIdNow(internalId: String): EkoChannelEntity? {
		return getByInternalIdNowImpl(internalId)
	}

	@Query("SELECT channel.*," +
			" channel_membership.readToSegment as membership_readToSegment," +
			" channel_extra.localReadToSegment as extra_localReadToSegment," +
			" channel_membership.lastMentionedSegment as membership_lastMentionedSegment," +
			" channel_extra.localLastMentionedSegment as extra_localLastMentionedSegment" +
			" from channel, channel_membership, channel_extra" +
			" where channel.channelId = channel_membership.channelId and channel.channelId = channel_extra.channelId" +
			" and channel.channelId  IN (:ids)")
	abstract fun getByIdsNowImpl(ids: List<String>): List<EkoChannelWithMembershipAndExtra>
	override fun getByIdsNow(ids: List<String>): List<EkoChannelEntity> {
		return FluentIterable.from(getByIdsNowImpl(ids))
				.transform { input : EkoChannelWithMembershipAndExtra? ->
					input
				}.toList().filterNotNull()
	}

	@Query("UPDATE channel set lastActivity = :lastActivity where channelId = :channelId")
	abstract fun updateLastActivity(channelId: String, lastActivity: DateTime)

	@Query("UPDATE channel set isMuted = :isMuted where channelId = :channelId")
	abstract fun updateIsMuted(channelId: String, isMuted: Boolean)

	// dummy update
	@Query("UPDATE channel set channelId = :channelId where channelId = :channelId")
	abstract fun notifyChanges(channelId: String)
	@Query("UPDATE channel set messageCount = :messageCount where channelId = :channelId and messageCount < :messageCount")
	abstract fun updateMessageCount(channelId: String, messageCount: Int)
	@Query("DELETE from channel")
	abstract override fun deleteAll()
	@Query("DELETE from channel where channelId = :channelId")
	abstract fun deleteByIdImpl(channelId: String)
	@Transaction
	open fun deleteById(channelId: String) {
		deleteByIdImpl(channelId)
		messageDao.deleteAllFromChannel(channelId)
		channelMembershipDao.deleteAllFromChannel(channelId)
		channelExtraDao.deleteAllFromChannel(channelId)
	}

	@Query("SELECT channelId " +
			"from channel " +
			"where channelId in (SELECT channelId from channel_membership where membership = 'member' and userId = :userId)")
	abstract fun getActiveIds(userId: String): Flowable<List<String>>

	@Query("SELECT channelId" +
			" from channel" +
			" where channelId in (SELECT channelId from channel_membership where (membership = 'none' or membership = 'banned') and userId = :userId)")
	abstract fun getInactiveIds(userId: String): List<String>
	@Transaction
	open fun deleteAllLocallyInactiveChannelsAndUpdateAllActiveChannelsToNotReading() {
		val inactiveChannelIds = getInactiveIds(getUserId())
		for (inactiveChannelId in inactiveChannelIds) {
			deleteById(inactiveChannelId)
		}
		channelExtraDao.updateAllReadStatuses(EkoChannelReadStatus.NOT_READING)
	}

	private fun beforeInsertOrUpdate(channel: EkoChannelEntity) {
		val c = getByIdNow(channel.channelId)
		if (c != null) {
			channel.messageCount = Math.max(c.messageCount, channel.messageCount)
		}
	}

	private fun beforeInsertOrUpdate(cs: List<EkoChannelEntity>) {
		for (channel in cs) {
			beforeInsertOrUpdate(channel)
		}
	}

	@Transaction
	override fun insert(channel: EkoChannelEntity) {
		beforeInsertOrUpdate(channel)
		super.insert(channel)
		update(channel, channelTagDao, EkoChannelTag.factory)
		channelExtraDao.insertOrUpdate(channel.channelId)
	}

	@Transaction
	override fun insert(channels: List<EkoChannelEntity>) {
		beforeInsertOrUpdate(channels)
		super.insert(channels)
		update(channels, channelTagDao, EkoChannelTag.factory)
		channelExtraDao.insertOrUpdate(FluentIterable.from(channels)
				.transform { channel: EkoChannelEntity? -> channel?.channelId }.toList().filterNotNull())
	}

	@Transaction
	override fun update(channel: EkoChannelEntity) {
		beforeInsertOrUpdate(channel)
		super.update(channel)
		update(channel, channelTagDao, EkoChannelTag.factory)
		channelExtraDao.insertOrUpdate(channel.channelId)
	}

	@Query("UPDATE channel set channelMarkerHash = :hash where channelId = :channelId")
	abstract fun updateMarkerHash(channelId: String, hash: Int)

	@Query("UPDATE channel set messagePreviewId = :messagePreviewId where channelId = :channelId")
	abstract fun updateMessagePreview(channelId: String, messagePreviewId: String)

	@Query("SELECT messagePreviewId FROM  channel WHERE channelId = :channelId LIMIT 1")
	abstract fun getMessagePreviewId(channelId: String): String?

	@Query("UPDATE channel set channelMarkerHash = :hash where channelId in (:channelIds)")
	abstract fun notifyChannelsChanges(channelIds: List<String>, hash: Int)
}