package com.ekoapp.ekosdk.internal.data.dao;

import androidx.annotation.Nullable;
import androidx.room.Dao;
import androidx.room.Delete;
import androidx.room.Insert;
import androidx.room.OnConflictStrategy;
import androidx.room.Query;
import androidx.room.Transaction;
import androidx.room.Update;

import com.amity.socialcloud.sdk.model.core.user.AmityUserType;
import com.ekoapp.ekosdk.internal.data.UserDatabase;
import com.ekoapp.ekosdk.internal.data.model.EkoAccount;

import java.util.List;

import io.reactivex.rxjava3.core.Flowable;
import io.reactivex.rxjava3.core.Single;

@Dao
public abstract class EkoAccountDao {

    @Query("SELECT * from account")
    public abstract Flowable<List<EkoAccount>> getAll();

    public Single<EkoAccount> getCurrentAccountSingle() {
        return getCurrentAccountFlowableImpl().firstOrError();
    }

    @Query("SELECT * from account  LIMIT 1")
    abstract Flowable<EkoAccount> getCurrentAccountFlowableImpl();

    @Query("SELECT * from account LIMIT 1")
    public abstract EkoAccount getCurrentAccountNow();

    @Insert(onConflict = OnConflictStrategy.REPLACE)
    abstract void insert(EkoAccount ekoAccount);

    @Update(onConflict = OnConflictStrategy.REPLACE)
    public abstract void update(EkoAccount ekoAccount);

    @Transaction
    public EkoAccount activateAccount(String userId) {
        return createAccountIfNeed(userId);
    }

    @Transaction
    public EkoAccount activateVisitor() {
        return createVisitorAccountIfNeed();
    }

    @Transaction
    public EkoAccount deactivateAccount (String userId) {
        return createAccountIfNeed(userId);
    }

    @Transaction
    public void logoutAccount (String userId) {
        UserDatabase.get().clearAllTables();
        deleteAll();
    }

    private EkoAccount createAccountIfNeed(String userId) {
        EkoAccount account = getByIdNow(userId);
        if (account == null) {
            account = EkoAccount.create(userId);
            insert(account);
        }
        return account;
    }

    private EkoAccount createVisitorAccountIfNeed() {
        EkoAccount account = getByUserTypeNow(AmityUserType.VISITOR.getValue());
        if (account == null) {
            account = EkoAccount.create();
            insert(account);
        }
        return account;
    }

    @Delete
    public abstract void delete(EkoAccount account);

    @Nullable
    @Query("SELECT * from account where userId = :userId LIMIT 1")
    abstract EkoAccount getByIdNow(String userId);

    @Nullable
    @Query("SELECT * from account where userType = :userType LIMIT 1")
    abstract EkoAccount getByUserTypeNow(String userType);

    @Query("DELETE from account")
    public abstract void deleteAll();
}
