package com.ekoapp.ekosdk.internal

import androidx.paging.ExperimentalPagingApi
import androidx.paging.LoadType
import androidx.paging.PagingState
import androidx.paging.rxjava2.RxRemoteMediator
import com.ekoapp.ekosdk.internal.data.dao.AmityPagingTokenDao
import com.ekoapp.ekosdk.internal.data.model.EkoQueryToken
import io.reactivex.Maybe
import io.reactivex.MaybeTransformer
import io.reactivex.Single
import io.reactivex.schedulers.Schedulers
import kotlin.math.ceil
import kotlin.math.max

@OptIn(ExperimentalPagingApi::class)
abstract class AmityRxRemoteMediator<ENTITY : Any, TOKEN : EkoQueryToken, TOKEN_DAO : AmityPagingTokenDao<TOKEN>>(private val tokenDao: TOKEN_DAO) :
    RxRemoteMediator<Int, ENTITY>() {

    final override fun loadSingle(loadType: LoadType, state: PagingState<Int, ENTITY>): Single<MediatorResult> {
        val pageSize = state.config.pageSize

        return when (loadType) {
            LoadType.REFRESH -> {
                state.anchorPosition?.let { anchorPosition ->
                    val pageNumber = ceil(max(1, anchorPosition).toDouble() / state.config.pageSize.toDouble()).toInt()
                    loadPage(pageNumber = pageNumber, pageSize = pageSize)
                        .subscribeOn(Schedulers.io())
                        .flatMap { insertToken(applyPrimaryKeysToToken(it)) }
                        .compose(interceptErrorAndEmpty)
                        .toSingle()
                } ?: run {
                    loadFirstPage(pageSize = pageSize)
                        .subscribeOn(Schedulers.io())
                        .flatMap { insertToken(applyPrimaryKeysToToken(it)) }
                        .compose(interceptErrorAndEmpty)
                        .toSingle()
                }
            }
            LoadType.PREPEND -> {
                if (stackFromEnd()) {
                    tokenDao.getFirstQueryToken(primaryKeys = primaryKeys())
                        .subscribeOn(Schedulers.io())
                        .flatMap { loadPreviousPage(token = it, pageSize = pageSize) }
                        .flatMap { insertToken(applyPrimaryKeysToToken(it)) }
                        .compose(interceptErrorAndEmpty)
                        .toSingle()
                } else {
                    Single.just<MediatorResult>(MediatorResult.Success(false))
                }
            }
            LoadType.APPEND -> {
                if (stackFromEnd()) {
                    Single.just<MediatorResult>(MediatorResult.Success(false))
                } else {
                    tokenDao.getLastQueryToken(primaryKeys = primaryKeys())
                        .subscribeOn(Schedulers.io())
                        .flatMap { loadNextPage(token = it, pageSize = pageSize) }
                        .flatMap { insertToken(applyPrimaryKeysToToken(it)) }
                        .compose(interceptErrorAndEmpty)
                        .toSingle()
                }
            }
        }
    }

    abstract fun loadPage(pageNumber: Int, pageSize: Int): Maybe<TOKEN>

    // can be either a first page from top or bottom depends on stackFromEnd()
    abstract fun loadFirstPage(pageSize: Int): Maybe<TOKEN>

    abstract fun loadNextPage(token: TOKEN, pageSize: Int): Maybe<TOKEN>

    open fun loadPreviousPage(token: TOKEN, pageSize: Int): Maybe<TOKEN> {
        return if (stackFromEnd()) {
            Maybe.error<TOKEN>(Exception("Stack from end-able MUST implement previous();"))
        } else {
            Maybe.never()
        }
    }

    private fun insertToken(token: TOKEN): Maybe<MediatorResult> {
        return if (token.pageNumber == EkoQueryToken.INVALID_PAGE_NUMBER) {
            Maybe.error<MediatorResult>(Exception("Invalid page number! please make sure that you specify a correct page number for a token"))
        } else {
            val isLastPage = if (stackFromEnd()) token.previous == null else token.next == null
            if (isLastPage) {
                tokenDao.deleteObsoleteQueryToken(primaryKeys(), token.pageNumber)
            }
            tokenDao.insertToken(token)
                .andThen(Maybe.just<MediatorResult>(MediatorResult.Success(isLastPage)))
        }
    }

    private val interceptErrorAndEmpty = MaybeTransformer<MediatorResult, MediatorResult> { upstream ->
        upstream.onErrorReturn { MediatorResult.Error(it) }
            .switchIfEmpty(Maybe.just(MediatorResult.Success(true)))
    }

    abstract fun primaryKeys(): Map<String, Any>

    abstract fun applyPrimaryKeysToToken(token: TOKEN): TOKEN

    abstract fun stackFromEnd(): Boolean
}