package com.amity.socialcloud.sdk.social.data.post

import com.amity.socialcloud.sdk.model.core.mention.AmityMentioneeTarget
import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.amity.socialcloud.sdk.model.social.post.AmityPostAttachment
import com.ekoapp.ekosdk.internal.api.AmityHttpClient
import com.ekoapp.ekosdk.internal.api.dto.EkoDeletionDto
import com.ekoapp.ekosdk.internal.api.dto.EkoPostQueryDto
import com.ekoapp.ekosdk.internal.api.socket.request.CreatePostV4Request
import com.ekoapp.ekosdk.internal.api.socket.request.FlagContentRequest
import com.ekoapp.ekosdk.internal.api.socket.request.PostUpdateRequest
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Single

internal class PostRemoteDataStore {

    fun fetchPost(postId: String): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.getPost(postId)
        }
    }

    fun approvePost(postId: String): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.approvePost(postId)
        }
    }

    fun declinePost(postId: String): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.declinePost(postId)
        }
    }

    fun deletePost(postId: String, hardDelete: Boolean): Single<EkoDeletionDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.deletePost(postId = postId, permanent = hardDelete)
        }
    }

    fun createPost(
        targetType: String, targetId: String,
        data: JsonObject,
        attachments: List<AmityPostAttachment>?,
        dataType: AmityPost.DataType? = null,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>?,
        hashtags: List<String>?,
    ): Single<EkoPostQueryDto> {
        val createPostRequest = CreatePostV4Request(
            targetType,
            targetId,
            data,
            dataType?.getApiKey(),
            attachments,
            metadata,
            mentionees,
            hashtags,
        )
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.createPost(createPostRequest)
        }
    }

    fun flagPost(postId: String, reason: FlagContentRequest): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.flagPost(postId, reason)
        }
    }

    fun unFlagPost(postId: String): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.unflagPost(postId)
        }
    }

    fun editPost(
        postId: String,
        data: JsonObject,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>?,
        attachments: List<AmityPostAttachment>?,
        hashtags: List<String>? = null,
    ): Single<EkoPostQueryDto> {
        val postUpdateRequest =
            PostUpdateRequest(postId, data, metadata, mentionees, attachments, hashtags)
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.updatePost(postId, postUpdateRequest)
        }
    }

    fun getCommunityPosts(
        targetId: String,
        sortBy: String,
        isDeleted: Boolean?,
        feedType: String?,
        dataTypes: List<String>? = null,
        matchingOnlyParentPost: Boolean = true,
        includeMixedStructure: Boolean,
        limit: Int? = null,
        token: String? = null,
    ): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.queryPosts(
                targetType = "community",
                targetId = targetId,
                isDeleted = isDeleted,
                feedType = feedType?.let { listOf(it) },
                dataTypes = dataTypes,
                sortBy = sortBy,
                matchingOnlyParentPost = matchingOnlyParentPost,
                includeMixedStructure = includeMixedStructure,
                limit = limit,
                token = token
            )
        }
    }

    fun getUserPosts(
        userId: String,
        isDeleted: Boolean?,
        postTypes: List<String>?,
        sortBy: String,
        matchingOnlyParentPost: Boolean = true,
        includeMixedStructure: Boolean,
        limit: Int? = null,
        token: String? = null,
    ): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.queryPosts(
                targetType = "user",
                targetId = userId,
                isDeleted = isDeleted,
                dataTypes = postTypes,
                sortBy = sortBy,
                matchingOnlyParentPost = matchingOnlyParentPost,
                includeMixedStructure = includeMixedStructure,
                limit = limit,
                token = token
            )
        }

    }

    fun getUserFeed(
        userId: String,
        isDeleted: Boolean?,
        postTypes: List<String>?,
        feedSources: List<String>,
        sortBy: String,
        matchingOnlyParentPost: Boolean = true,
        includeMixedStructure: Boolean,
        limit: Int? = null,
        token: String? = null,
    ): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.getUserFeed(
                userId = userId,
                isDeleted = isDeleted,
                dataTypes = postTypes,
                sortBy = sortBy,
                feedSources = feedSources,
                matchingOnlyParentPost = matchingOnlyParentPost,
                includeMixedStructure = includeMixedStructure,
                limit = limit,
                token = token
            )
        }
    }

    fun searchPosts(
        targetType: String?,
        targetId: String?,
        hashtags: List<String>? = null,
        isDeleted: Boolean?,
        feedType: String?,
        dataTypes: List<String>? = null,
        matchingOnlyParentPost: Boolean = true,
        includeMixedStructure: Boolean,
        limit: Int? = null,
        token: String? = null,
    ): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.searchPosts(
                targetType = targetType ?: "all",
                targetId = targetId,
                hashtags = hashtags,
                isDeleted = isDeleted,
                feedType = feedType?.let { listOf(it) },
                dataTypes = dataTypes,
                matchingOnlyParentPost = matchingOnlyParentPost,
                includeMixedStructure = includeMixedStructure,
                limit = limit,
                token = token
            )
        }
    }

    fun getGlobalFeedPosts(
        dataTypes: List<String>? = null,
        resolveParent: Boolean? = null,
        includeMixedStructure: Boolean,
        limit: Int? = null,
        token: String? = null,
    ): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.globalFeedV4(
                dataTypes = dataTypes,
                resolveParent = resolveParent,
                includeMixedStructure = includeMixedStructure,
                limit = limit,
                token = token
            )
        }
    }

    fun getCustomRankingPosts(
        includeMixedStructure: Boolean,
        limit: Int? = null,
        token: String? = null
    ): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.globalFeedV5(
                includeMixedStructure = includeMixedStructure,
                limit = limit,
                token = token
            )
        }
    }

    fun getPostByIds(postIds: List<String>): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.getPostByIds(postIds)
        }
    }

    fun isFlaggedByMe(postId: String): Single<JsonObject> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.isFlaggedByMe(postId)
        }
    }

    fun semanticSearchPosts(
        query: String,
        targetType: String?,
        targetId: String?,
        dataTypes: List<String>? = null,
        matchingOnlyParentPost: Boolean = true,
        includeMixedStructure: Boolean,
        limit: Int? = null,
        token: String? = null,
    ): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.semanticSearchPosts(
                query = query,
                targetType = targetType,
                targetId = targetId.takeIf { targetId?.isNotEmpty() == true },
                dataTypes = dataTypes,
                matchingOnlyParentPost = matchingOnlyParentPost,
                includeMixedStructure = includeMixedStructure,
                limit = limit,
                token = token
            )
        }
    }

}