package com.amity.socialcloud.sdk.social.data.event

import com.amity.socialcloud.sdk.entity.social.event.EventEntity
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.schedulers.Schedulers
import io.reactivex.rxjava3.subjects.BehaviorSubject
import java.util.concurrent.ConcurrentHashMap

/**
 * Temporary in-memory data store for events until Room database implementation is available.
 * Uses RxJava BehaviorSubjects to provide reactive updates to observers.
 */
internal class EventLocalDataStore {

    companion object {
        // In-memory cache using thread-safe ConcurrentHashMap
        private val eventCache = ConcurrentHashMap<String, EventEntity>()
        
        // Subjects for reactive updates, one per event ID
        private val eventSubjects = ConcurrentHashMap<String, BehaviorSubject<EventEntity>>()
    }

    fun saveEvent(event: EventEntity): Completable {
        return Completable.fromAction {
            eventCache[event.eventId] = event
            
            // Notify observers
            val subject = eventSubjects.getOrPut(event.eventId) {
                BehaviorSubject.create<EventEntity>()
            }
            subject.onNext(event)
        }.subscribeOn(Schedulers.io())
    }

    fun saveEvents(events: List<EventEntity>): Completable {
        return Completable.fromAction {
            events.forEach { event ->
                eventCache[event.eventId] = event
                
                // Notify observers
                val subject = eventSubjects.getOrPut(event.eventId) {
                    BehaviorSubject.create<EventEntity>()
                }
                subject.onNext(event)
            }
        }.subscribeOn(Schedulers.io())
    }

    fun getEvent(eventId: String): EventEntity? {
        return eventCache[eventId]
    }

    fun observeEvent(eventId: String): Flowable<EventEntity> {
        return Flowable.defer {
            val subject = eventSubjects.getOrPut(eventId) {
                BehaviorSubject.create<EventEntity>()
            }
            
            // If we have cached data, emit it immediately
            eventCache[eventId]?.let { subject.onNext(it) }
            
            subject.toFlowable(io.reactivex.rxjava3.core.BackpressureStrategy.LATEST)
                .map {
                    if (it == null) {
                        throw AmityException.create(
                            message = "item not found",
                            cause = null,
                            error = AmityError.ITEM_NOT_FOUND
                        )
                    }
                    it
                }
        }.subscribeOn(Schedulers.io())
    }

    fun deleteEvent(eventId: String): Completable {
        return Completable.fromAction {
            eventCache.remove(eventId)
            eventSubjects[eventId]?.onComplete()
            eventSubjects.remove(eventId)
        }.subscribeOn(Schedulers.io())
    }

    fun clear(): Completable {
        return Completable.fromAction {
            eventCache.clear()
            eventSubjects.values.forEach { it.onComplete() }
            eventSubjects.clear()
        }.subscribeOn(Schedulers.io())
    }
}
